"""
Professional Zing Music Media Player Implementation

Auto-Advance Implementation Notes:
This implementation follows the professional patterns used by Home Assistant's 
official Cast integration (homeassistant/components/cast/media_player.py).

Key Professional Approaches:
1. IMMEDIATE RESPONSE: No arbitrary delays for auto-advance detection
2. STATE-DRIVEN: Direct response to media player state changes (idle/off)
3. SMART DETECTION: Cast device pattern recognition with position validation
4. EFFICIENT: Uses media_status callbacks without polling or delays

This matches how official Cast integrations handle track endings - they respond
immediately to device state changes rather than using timer-based delays.
"""

from datetime import datetime, timedelta, timezone
import logging
import time
import asyncio
import subprocess
import json
from typing import Optional, List
import aiohttp
import io

from homeassistant.components.media_player import (
    MediaPlayerEntity,
    MediaPlayerEntityFeature,
    MediaType,
    BrowseMedia,
    SearchMedia,
    SearchMediaQuery,
)
from homeassistant.helpers.event import async_call_later, async_track_state_change_event
from homeassistant.config_entries import ConfigEntry
from homeassistant.const import STATE_IDLE, STATE_PLAYING, STATE_PAUSED, STATE_OFF
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.dispatcher import async_dispatcher_connect
from homeassistant.exceptions import HomeAssistantError
from homeassistant.helpers import entity_registry as er, device_registry as dr

# Audio duration detection
try:
    from mutagen import File as MutagenFile
    MUTAGEN_AVAILABLE = True
except ImportError:
    MUTAGEN_AVAILABLE = False
    _LOGGER.warning("mutagen not available - will fallback to metadata duration")

from .const import (
    DOMAIN, MEDIA_TYPE_ARTIST, MEDIA_TYPE_ALBUM, MEDIA_TYPE_TRACK, 
    MEDIA_TYPE_MUSIC, MEDIA_TYPE_RECENT_TRACKS, MEDIA_TYPE_PODCAST, 
    MEDIA_TYPE_KIDS, MEDIA_TYPE_PLAYLIST, MEDIA_TYPE_FEATURED_PLAYLIST, 
    MEDIA_TYPE_ARTISTS_PLAYLIST, MEDIA_TYPE_LATEST, MEDIA_TYPE_LATEST_ALBUMS,
    MEDIA_TYPE_LATEST_SINGLES, MEDIA_TYPE_LATEST_SONGS, MEDIA_TYPE_ARTIST_ALBUMS,
    MEDIA_TYPE_ARTIST_SINGLES, MEDIA_TYPE_ARTIST_SONGS, MEDIA_TYPE_ARTIST_LATEST,
    MEDIA_TYPE_ARTIST_RELATED, MEDIA_TYPE_KIDS_ARTISTS, MEDIA_TYPE_KIDS_ARTIST,
    MEDIA_TYPE_KIDS_ARTIST_ALBUMS, MEDIA_TYPE_KIDS_ARTIST_LATEST,
    MEDIA_TYPE_PODCAST_HOSTS, MEDIA_TYPE_PODCAST_HOST,
    CONF_ALLOWED_ENTITY_IDS, CONF_KIDS_ONLY, CONF_LATEST_COUNT, SIGNAL_OPTIONS_UPDATED, ARTIST_SINGLES_COUNT, ARTIST_SONGS_COUNT
)
from .api import ZingMusicAPI, Track, PlaybackController

_LOGGER = logging.getLogger(__name__)

async def async_setup_entry(
    hass: HomeAssistant,
    config_entry: ConfigEntry,
    async_add_entities: AddEntitiesCallback,
) -> None:
    store = hass.data[DOMAIN][config_entry.entry_id]
    api: ZingMusicAPI = store["api"]
    async_add_entities([ZingMusicMediaPlayer(hass, api, config_entry.entry_id)])

class ZingMusicMediaPlayer(MediaPlayerEntity):
    _attr_name = "Zing Music Player"
    _attr_icon = "mdi:music-box-multiple-outline"
    _attr_media_content_type = MediaType.MUSIC
    _attr_supported_features = (
        MediaPlayerEntityFeature.BROWSE_MEDIA
        | MediaPlayerEntityFeature.PLAY_MEDIA
        | MediaPlayerEntityFeature.STOP
        | MediaPlayerEntityFeature.PAUSE
        | MediaPlayerEntityFeature.PLAY
        | MediaPlayerEntityFeature.SEARCH_MEDIA
        | MediaPlayerEntityFeature.SELECT_SOURCE
        | MediaPlayerEntityFeature.SEEK
        | MediaPlayerEntityFeature.NEXT_TRACK
        | MediaPlayerEntityFeature.PREVIOUS_TRACK
        | MediaPlayerEntityFeature.REPEAT_SET
        | MediaPlayerEntityFeature.SHUFFLE_SET
        | MediaPlayerEntityFeature.VOLUME_SET
        | MediaPlayerEntityFeature.VOLUME_MUTE
        | MediaPlayerEntityFeature.VOLUME_STEP
        | MediaPlayerEntityFeature.TURN_ON
        | MediaPlayerEntityFeature.TURN_OFF
    )

    def __init__(self, hass: HomeAssistant, api: ZingMusicAPI, entry_id: str) -> None:
        self.hass = hass
        self._api = api
        self._entry_id = entry_id
        self._options_unsub = async_dispatcher_connect(
            self.hass,
            f"{SIGNAL_OPTIONS_UPDATED}_{self._entry_id}",
            self._handle_options_updated,
        )
        self._state = STATE_IDLE
        self._current_track: Optional[Track] = None
        self._media_image_url: Optional[str] = None
        self._media_artist: Optional[str] = None
        self._media_title: Optional[str] = None
        self._media_album_name: Optional[str] = None
        self._search_result: Optional[BrowseMedia] = None

        # Cache Cast device detection to avoid blocking during state changes
        self._is_cast_device_cache: dict[str, bool] = {}

        # Source selection:
        # - _attr_source stores the REAL entity_id (used for service calls)
        # - _source_label is the friendly display name shown in the UI dropdown
        self._attr_source: Optional[str] = None
        self._source_label: Optional[str] = None
        self._source_map: dict[str, str] = {}  # label -> entity_id

        self._media_position = 0.0
        self._media_position_updated_at: Optional[datetime] = None
        self._media_url: Optional[str] = None
        self._media_duration: Optional[float] = None
        
        # Track when a new track starts playing to prevent premature auto-advance
        self._track_start_time: Optional[datetime] = None

        # source monitoring
        self._unsub_source_listener = None

        # playback behavior
        self._repeat: str = "off"   # "off" | "one" | "all"
        self._shuffle: bool = False

        # volume control (delegated to source)
        self._volume_level: Optional[float] = None
        self._is_volume_muted: Optional[bool] = None

        self._playback_controller = PlaybackController(api)

        # Language selection from config entry options
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        from .const import CONF_LANGUAGE, LANG_ENGLISH
        self._language = (entry.options.get(CONF_LANGUAGE) if entry and entry.options else LANG_ENGLISH)
        _LOGGER.debug(f"Zing Music Player initialized. Language: {self._language}")

    async def async_will_remove_from_hass(self) -> None:
        if getattr(self, "_options_unsub", None):
            self._options_unsub()
            self._options_unsub = None

    def _allowed_entity_ids(self) -> Optional[List[str]]:
        """Return the allowed entity_ids from hass.data if present.
        Falls back to None if not configured (means allow all)."""
        store = self.hass.data.get(DOMAIN, {}).get(self._entry_id, {})
        allowed = store.get("allowed_entity_ids")
        return allowed

    def _restricted_artists(self) -> List[str]:
        """Return the list of restricted artist names from config options."""
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        from .const import CONF_RESTRICTED_ARTISTS
        if entry and entry.options:
            raw_restricted = entry.options.get(CONF_RESTRICTED_ARTISTS, [])
            
            # Handle different data types that might be stored
            if isinstance(raw_restricted, list):
                filtered_result = [name.strip() for name in raw_restricted if name and name.strip()]
                return filtered_result
            elif isinstance(raw_restricted, str):
                if raw_restricted.strip():
                    str_result = [name.strip() for name in raw_restricted.split(",") if name.strip()]
                    return str_result
                else:
                    return []
            else:
                return []
        return []

    def _whitelisted_artists(self) -> List[str]:
        """Return the list of whitelisted artist names from config options."""
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        from .const import CONF_WHITELISTED_ARTISTS
        if entry and entry.options:
            raw_whitelisted = entry.options.get(CONF_WHITELISTED_ARTISTS, [])
            
            # Handle different data types that might be stored
            if isinstance(raw_whitelisted, list):
                filtered_result = [name.strip() for name in raw_whitelisted if name and name.strip()]
                return filtered_result
            elif isinstance(raw_whitelisted, str):
                if raw_whitelisted.strip():
                    str_result = [name.strip() for name in raw_whitelisted.split(",") if name.strip()]
                    return str_result
                else:
                    return []
            else:
                return []
        return []

    def _is_kids_only_mode(self) -> bool:
        """Return True if kids-only mode is enabled in config options."""
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        if entry and entry.options:
            return entry.options.get(CONF_KIDS_ONLY, False)
        return False

    def _get_latest_count(self) -> int:
        """Return the configured count for latest items."""
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        if entry and entry.options:
            from .const import CONF_LATEST_COUNT, DEFAULT_LATEST_COUNT
            return entry.options.get(CONF_LATEST_COUNT, DEFAULT_LATEST_COUNT)
        from .const import DEFAULT_LATEST_COUNT
        return DEFAULT_LATEST_COUNT

    def _get_display_name(self, obj):
        """Get the appropriate display name based on selected language."""
        # Update language setting from config entry options
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        from .const import CONF_LANGUAGE, LANG_HEBREW
        current_language = entry.options.get(CONF_LANGUAGE) if entry and entry.options else None
        
        # Check for Hebrew (case-insensitive)
        is_hebrew = current_language and current_language.lower() == "hebrew"
        
        if is_hebrew:
            return getattr(obj, "heName", None) or getattr(obj, "enName", None)
        else:
            return getattr(obj, "enName", None) or getattr(obj, "heName", None)

    def _is_artist_allowed(self, obj) -> bool:
        """Check if an artist should be shown based on whitelist and blacklist rules."""
        # Get whitelist and blacklist
        whitelisted = self._whitelisted_artists()
        restricted = self._restricted_artists()
        
        # Function to check if artist name matches any in a list
        def matches_any_in_list(artist_names, name_list):
            if not name_list:
                return False
            name_list_lower = [name.lower().strip() for name in name_list if name.strip()]
            if not name_list_lower:
                return False
            
            for artist_name in artist_names:
                if not artist_name:
                    continue
                artist_name_lower = artist_name.lower().strip()
                for list_name in name_list_lower:
                    if list_name in artist_name_lower or artist_name_lower in list_name:
                        return True
            return False
        
        # Get all artist names for this object
        artist_names = []
        
        # Check if the object itself is an artist
        if hasattr(obj, 'enName') or hasattr(obj, 'heName'):
            en_name = getattr(obj, 'enName', '') or ''
            he_name = getattr(obj, 'heName', '') or ''
            if en_name:
                artist_names.append(en_name)
            if he_name:
                artist_names.append(he_name)
        
        # Check associated artists
        artists_to_check = []
        if hasattr(obj, 'artists') and obj.artists:
            artists_to_check.extend(obj.artists)
        if hasattr(obj, 'featuredArtists') and obj.featuredArtists:
            artists_to_check.extend(obj.featuredArtists)
        if hasattr(obj, 'album') and obj.album:
            if hasattr(obj.album, 'artists') and obj.album.artists:
                artists_to_check.extend(obj.album.artists)
            if hasattr(obj.album, 'featuredArtists') and obj.album.featuredArtists:
                artists_to_check.extend(obj.album.featuredArtists)
                
        for artist in artists_to_check:
            en_name = getattr(artist, 'enName', '') or ''
            he_name = getattr(artist, 'heName', '') or ''
            if en_name:
                artist_names.append(en_name)
            if he_name:
                artist_names.append(he_name)
        
        # If no artist names found, reject
        if not artist_names:
            return False
        
        # If whitelist is not empty, only show artists in whitelist
        if whitelisted:
            if not matches_any_in_list(artist_names, whitelisted):
                return False
        
        # Always check blacklist (restricted artists)
        if restricted:
            if matches_any_in_list(artist_names, restricted):
                return False
        
        return True

    async def _get_duration_with_ffprobe(self, audio_url: str) -> Optional[float]:
        """Get accurate duration using FFprobe."""
        try:
            
            # Construct the FFprobe command
            cmd = [
                'ffprobe',
                '-v', 'error',
                '-show_entries', 'format=duration',
                '-of', 'json',
                audio_url
            ]
            
            # Run FFprobe asynchronously with timeout
            process = await asyncio.create_subprocess_exec(
                *cmd,
                stdout=asyncio.subprocess.PIPE,
                stderr=asyncio.subprocess.PIPE
            )
            
            try:
                stdout, stderr = await asyncio.wait_for(process.communicate(), timeout=15.0)
            except asyncio.TimeoutError:
                _LOGGER.warning(f"[ZING] FFprobe timeout for {audio_url}")
                process.kill()
                return None
            
            if process.returncode != 0:
                return None
            
            # Parse JSON output
            try:
                data = json.loads(stdout.decode())
                duration = float(data['format']['duration'])
                _LOGGER.info(f"[ZING] FFprobe detected duration: {duration:.2f}s")
                return duration
            except (json.JSONDecodeError, KeyError, ValueError) as e:
                _LOGGER.debug(f"[ZING] Failed to parse FFprobe output: {e}")
                return None
                
        except FileNotFoundError:
            _LOGGER.debug("[ZING] FFprobe not available in system")
            return None
        except Exception as e:
            _LOGGER.debug(f"[ZING] FFprobe detection failed: {e}")
            return None

    async def _get_actual_audio_duration(self, audio_url: str) -> Optional[float]:
        """Get the actual duration from the audio file, not metadata."""
        # Try FFprobe first (most accurate)
        ffprobe_duration = await self._get_duration_with_ffprobe(audio_url)
        if ffprobe_duration:
            return ffprobe_duration
        
        # Fallback to mutagen + file size estimation
        if not MUTAGEN_AVAILABLE:
            _LOGGER.debug("[ZING] FFprobe failed and mutagen not available, skipping audio duration detection")
            return None
            
        try:
            _LOGGER.debug(f"[ZING] FFprobe failed, trying mutagen duration detection for: {audio_url}")
            
            # Download a portion of the file to analyze
            timeout = aiohttp.ClientTimeout(total=30)
            async with aiohttp.ClientSession(timeout=timeout) as session:
                # First, get file size for fallback calculation
                file_size = None
                try:
                    async with session.head(audio_url) as head_response:
                        if head_response.status == 200:
                            content_length = head_response.headers.get('content-length')
                            if content_length:
                                file_size = int(content_length)
                except Exception as e:
                    _LOGGER.debug(f"[ZING] Could not get file size: {e}")
                
                # Try multiple approaches for duration detection
                duration = None
                
                # Approach 1: Try with larger initial chunk (2MB)
                headers = {'Range': 'bytes=0-2097152'}  # First 2MB
                try:
                    async with session.get(audio_url, headers=headers) as response:
                        if response.status in [200, 206]:  # 206 = Partial Content
                            data = await response.read()
                            audio_file = io.BytesIO(data)
                            
                            mutagen_file = MutagenFile(audio_file)
                            if mutagen_file and hasattr(mutagen_file, 'info') and mutagen_file.info:
                                duration = getattr(mutagen_file.info, 'length', None)
                                if duration and duration > 0:
                                    # Validate header-based duration against file size
                                    if file_size:
                                        # Smart file size estimation based on file size patterns
                                        file_size_mb = file_size / (1024 * 1024)
                                        if file_size_mb > 8:
                                            estimated_kbps = 256
                                        elif file_size_mb > 5:
                                            estimated_kbps = 192
                                        elif file_size_mb > 3:  # Our 3.88MB file
                                            estimated_kbps = 160  # Should give ~241s
                                        elif file_size_mb > 2:
                                            estimated_kbps = 128
                                        else:
                                            estimated_kbps = 96
                                        
                                        estimated_duration = (file_size * 8) / (estimated_kbps * 1000)
                                        ratio = duration / estimated_duration if estimated_duration > 0 else 0
                                        
                                        # For VBR MP3s, prefer file size estimate if header is significantly off
                                        # Tighten validation - if header is >25% off from size estimate, use size
                                        if 0.75 < ratio < 1.35:  # Header duration is close to size estimate
                                            _LOGGER.info(f"[ZING] Header duration {duration:.2f}s validated (ratio {ratio:.2f} vs size estimate {estimated_duration:.2f}s)")
                                            return float(duration)
                                        else:
                                            _LOGGER.info(f"[ZING] Header duration ({duration:.2f}s) ratio {ratio:.2f} too far from file size estimate ({estimated_duration:.2f}s), preferring size estimate")
                                            return float(estimated_duration)
                                    else:
                                        _LOGGER.info(f"[ZING] Detected duration from 2MB header: {duration:.2f}s")
                                        return float(duration)
                except Exception as e:
                    _LOGGER.debug(f"[ZING] 2MB header analysis failed: {e}")
                
                # Approach 2: For MP3 files, use HTTP range sampling to estimate average bitrate
                if not duration and file_size and audio_url.lower().endswith('.mp3'):
                    try:
                        # Sample bitrate from multiple parts of the file for VBR accuracy
                        sample_size = 32768  # 32KB samples
                        samples_to_take = min(5, file_size // (sample_size * 4))  # Up to 5 samples
                        
                        if samples_to_take >= 2:
                            total_sample_duration = 0
                            total_sample_size = 0
                            
                            for i in range(samples_to_take):
                                # Sample from different positions in the file
                                start_pos = (file_size // samples_to_take) * i
                                end_pos = min(start_pos + sample_size - 1, file_size - 1)
                                
                                headers = {'Range': f'bytes={start_pos}-{end_pos}'}
                                async with session.get(audio_url, headers=headers) as response:
                                    if response.status in [200, 206]:
                                        sample_data = await response.read()
                                        sample_file = io.BytesIO(sample_data)
                                        
                                        try:
                                            mutagen_file = MutagenFile(sample_file)
                                            if mutagen_file and hasattr(mutagen_file, 'info') and mutagen_file.info:
                                                bitrate = getattr(mutagen_file.info, 'bitrate', None)
                                                if bitrate and bitrate > 0:
                                                    # Estimate duration for this sample
                                                    sample_duration = (len(sample_data) * 8) / bitrate
                                                    total_sample_duration += sample_duration
                                                    total_sample_size += len(sample_data)
                                        except Exception:
                                            continue
                            
                            # Calculate average bitrate and estimate total duration
                            if total_sample_duration > 0 and total_sample_size > 0:
                                avg_bytes_per_second = total_sample_size / total_sample_duration
                                duration = file_size / avg_bytes_per_second
                                
                                if 30 < duration < 7200:  # Reasonable range
                                    _LOGGER.info(f"[ZING] Multi-sample bitrate estimation: {duration:.2f}s")
                                    return float(duration)
                                    
                    except Exception as e:
                        _LOGGER.debug(f"[ZING] Multi-sample estimation failed: {e}")
                
                # Approach 3: Conservative file size estimate (last resort)
                if not duration and file_size:
                    # Improved file size estimation with better bitrate guessing
                    if audio_url.lower().endswith('.mp3'):
                        # For MP3, estimate based on file size patterns
                        file_size_mb = file_size / (1024 * 1024)
                        
                        if file_size_mb > 8:    # > 8MB
                            estimated_kbps = 256  # High quality
                        elif file_size_mb > 5:  # > 5MB  
                            estimated_kbps = 192  # Good quality
                        elif file_size_mb > 3:  # > 3MB (like our 3.88MB file)
                            estimated_kbps = 160  # Standard quality
                        elif file_size_mb > 2:  # > 2MB
                            estimated_kbps = 128  # Lower quality
                        else:
                            estimated_kbps = 96   # Very low quality
                    else:
                        # For other formats, use conservative estimate
                        estimated_kbps = 128
                    
                    # Calculate duration: file_size_bits / (bitrate_kbps * 1000)
                    estimated_duration = (file_size * 8) / (estimated_kbps * 1000)
                    
                    if 30 < estimated_duration < 7200:  # Reasonable range (30s to 2h)
                        _LOGGER.info(f"[ZING] File size estimate: {estimated_duration:.2f}s (assumed {estimated_kbps}kbps for {file_size_mb:.1f}MB file)")
                        return float(estimated_duration)
                                
        except Exception as e:
            _LOGGER.warning(f"[ZING] Failed to detect audio duration for {audio_url}: {e}")
            
        return None

    def _handle_options_updated(self) -> None:
        # Update language setting from new options
        entry = self.hass.config_entries.async_get_entry(self._entry_id)
        from .const import CONF_LANGUAGE, LANG_ENGLISH
        self._language = (entry.options.get(CONF_LANGUAGE) if entry and entry.options else LANG_ENGLISH)
        _LOGGER.debug(f"[ZING] Language updated to: {self._language}")
        
        # Log kids-only setting for debugging
        kids_only = self._is_kids_only_mode()
        
        # Log latest count setting for debugging
        latest_count = self._get_latest_count()
        
        # Log restricted and whitelisted artists for debugging
        restricted = self._restricted_artists()
        whitelisted = self._whitelisted_artists()

        # Rebuild the source map based on new options
        self._source_map = self._build_source_map()
        # If current selected entity_id is no longer allowed, clear selection
        allowed = set(self._allowed_entity_ids() or [])
        if self._attr_source and allowed and self._attr_source not in allowed:
            _LOGGER.debug(
                "[ZING] Current source %s no longer allowed; clearing selection",
                self._attr_source,
            )
            self._attr_source = None
            self._source_label = None
            self._detach_source_listener()
        
        # Schedule state update on the event loop to avoid threading issues
        self.hass.loop.call_soon_threadsafe(
            lambda: self.schedule_update_ha_state()
        )

    def _detach_source_listener(self):
        if self._unsub_source_listener:
            self._unsub_source_listener()
            self._unsub_source_listener = None

    def _log_now_playing(self, where: str = ""):
        _LOGGER.debug(
            "[ZING][%s] state=%s title=%s artist=%s album=%s img=%s pos=%s dur=%s updated_at=%s url=%s",
            where,
            getattr(self, "_state", None),
            getattr(self, "_media_title", None),
            getattr(self, "_media_artist", None),
            getattr(self, "_media_album_name", None),
            getattr(self, "_media_image_url", None),
            (self.media_position if hasattr(self, "media_position") else None),
            getattr(self, "_media_duration", None),
            getattr(self, "_media_position_updated_at", None),
            getattr(self, "_media_url", None),
        )

    def _build_source_map(self) -> dict[str, str]:
        """Map display label -> entity_id. Filter out unavailable and unsupported players."""
        states = self.hass.states.async_all("media_player")
        allowed = set(self._allowed_entity_ids() or [])
        
        _LOGGER.info(f"[ZING] _build_source_map: Found {len(states)} total media_player entities")
        _LOGGER.info(f"[ZING] _build_source_map: Allowed entities filter: {allowed}")
        
        # Get Music Assistant integration entities to exclude
        music_assistant_entities = set()
        try:
            import homeassistant.helpers.entity_registry as er
            entity_registry = er.async_get(self.hass)
            for entity in entity_registry.entities.values():
                if entity.platform == "music_assistant":
                    music_assistant_entities.add(entity.entity_id)
        except Exception:
            pass
        
        _LOGGER.debug(f"[ZING] _build_source_map: Music Assistant entities to exclude: {music_assistant_entities}")
        
        name_to_entities: dict[str, list[str]] = {}
        for st in states:
            # Log details for speakers_group specifically
            if st.entity_id == "media_player.speakers_group":
                _LOGGER.info(f"[ZING] Found speakers_group: state={st.state}, attributes={st.attributes}")
            
            # Filter out Music Assistant entities (by platform)
            if st.entity_id in music_assistant_entities:
                continue
            
            # Filter out Music Assistant entities (by app_id)
            if st.attributes.get('app_id') == 'music_assistant':
                continue
                
            # Filter out unavailable/unknown states - these can't be used for output
            # Exception: browser_mod entities can work in 'unknown' state
            if st.state in ['unavailable', 'unknown']:
                if "browser_mod" in st.entity_id:
                    if st.state == 'unavailable':
                        continue
                    # Allow 'unknown' state for browser_mod entities
                else:
                    continue
                
            # Check if media player supports play_media (bit 512)
            # This ensures the player can actually receive and play content from Zing
            supported_features = st.attributes.get('supported_features', 0)
            if not (supported_features & 512):  # MediaPlayerEntityFeature.PLAY_MEDIA = 512
                continue
                
            # Apply allowed entity filter if configured
            if allowed and st.entity_id not in allowed:
                continue
                
            # Use clean display name with group indicator if applicable
            display = st.name or st.entity_id
            
            # Add group indicator to display name for clarity
            if self._is_speaker_group(st.entity_id):
                group_members = st.attributes.get("entity_id", [])
                if len(group_members) > 1:
                    display += f" (Group of {len(group_members)})"
                elif st.attributes.get("group_members"):
                    # Handle other group types (Sonos, Cast, etc.)
                    group_members = st.attributes.get("group_members", [])
                    if len(group_members) > 1:
                        display += f" (Group of {len(group_members)})"
            
            name_to_entities.setdefault(display, []).append(st.entity_id)

        label_to_entity: dict[str, str] = {}
        for name, eids in name_to_entities.items():
            if len(eids) == 1:
                label_to_entity[name] = eids[0]
            else:
                # duplicate names → include entity_id
                for eid in eids:
                    label_to_entity[f"{name} ({eid})"] = eid
        
        _LOGGER.debug(f"[ZING] _build_source_map: Final source map: {label_to_entity}")
        return label_to_entity

    # ---------- standard properties ----------

    @property
    def unique_id(self) -> str:
        return f"{DOMAIN}_{self._entry_id}"

    @property
    def state(self) -> str:
        return self._state

    @property
    def media_image_url(self) -> Optional[str]:
        return self._media_image_url

    @property
    def media_artist(self) -> Optional[str]:
        return self._media_artist

    @property
    def media_title(self) -> Optional[str]:
        return self._media_title

    @property
    def media_album_name(self) -> Optional[str]:
        return self._media_album_name

    @property
    def entity_picture(self) -> Optional[str]:
        """Compatibility property used by many cards for artwork."""
        return self._media_image_url

    @property
    def source_list(self) -> list[str]:
        # Return FRIENDLY LABELS in the dropdown
        if not self._source_map:
            self._source_map = self._build_source_map()
        return sorted(self._source_map.keys(), key=lambda s: s.casefold())

    @property
    def source(self) -> Optional[str]:
        # Show the label; internally we keep entity_id in _attr_source
        if self._source_label:
            return self._source_label
        if not self._attr_source:
            return None
        st = self.hass.states.get(self._attr_source)
        return (st.name if st else self._attr_source)

    @property
    def media_position(self) -> Optional[float]:
        # Professional Cast Integration Approach: Always trust the source device's position
        # This matches exactly how the official Home Assistant Cast integration works
        if self._attr_source:
            source_state = self.hass.states.get(self._attr_source)
            if source_state:
                source_position = source_state.attributes.get("media_position")
                if source_position is not None:
                    try:
                        # Return the source's position directly - no calculation needed
                        return float(source_position)
                    except (ValueError, TypeError):
                        pass
        
        # Fallback for non-source playback or when source position is unavailable
        if self._state != STATE_PLAYING:
            return self._media_position
            
        # Only calculate elapsed time as last resort
        if not self._media_position_updated_at:
            return self._media_position
            
        delta = datetime.now(timezone.utc) - self._media_position_updated_at
        calculated_position = self._media_position + max(delta.total_seconds(), 0)
        
        # Cap at duration if we have it
        if self._media_duration:
            calculated_position = min(calculated_position, self._media_duration)
            
        return calculated_position

    @property
    def media_position_updated_at(self) -> Optional[datetime]:
        # Professional Cast Integration Approach: Use source's position timestamp
        if self._attr_source:
            source_state = self.hass.states.get(self._attr_source)
            if source_state:
                source_updated_at = source_state.attributes.get("media_position_updated_at")
                if source_updated_at:
                    return source_updated_at
        
        # Fallback to our own timestamp
        return self._media_position_updated_at

    @property
    def extra_state_attributes(self) -> dict:
        """Expose rich attributes for cards that read from attributes directly."""
        playlist = getattr(self._playback_controller, "playlist", None)
        playlist_size = len(playlist) if playlist else None
        playlist_index = (self._playback_controller.current_index + 1) if playlist else None
        
        # Get lyrics and credits from current track if available
        en_lyrics = None
        he_lyrics = None
        credits = None
        
        if self._current_track:
            en_lyrics = getattr(self._current_track, "enLyrics", None)
            he_lyrics = getattr(self._current_track, "heLyrics", None)
            
            # Format credits as a readable string
            credits_list = getattr(self._current_track, "credits", None)
            if credits_list and isinstance(credits_list, list):
                credit_strings = []
                for credit in credits_list:
                    if hasattr(credit, 'role') and hasattr(credit, 'enName'):
                        name = getattr(credit, 'enName', None) or getattr(credit, 'heName', None)
                        role = getattr(credit, 'role', None)
                        if name and role:
                            credit_strings.append(f"{name} ({role})")
                        elif name:
                            credit_strings.append(name)
                credits = "; ".join(credit_strings) if credit_strings else None
        
        return {
            # backend/media info
            "media_duration_mirrored": getattr(self, "_media_duration", None),
            "playlist_index": playlist_index,
            "playlist_size": playlist_size,
            "repeat": getattr(self, "_repeat", None),
            "shuffle": getattr(self, "_shuffle", None),

            # volume info
            "volume_level": getattr(self, "_volume_level", None),
            "is_volume_muted": getattr(self, "_is_volume_muted", None),

            # HA-standard names mirrored (some cards read attributes directly)
            "media_title": getattr(self, "_media_title", None),
            "media_artist": getattr(self, "_media_artist", None),
            "media_album_name": getattr(self, "_media_album_name", None),
            "media_image_url": getattr(self, "_media_image_url", None),
            "media_content_type": getattr(self, "_attr_media_content_type", "music"),

            # Lyrics and credits
            "en_lyrics": en_lyrics,
            "he_lyrics": he_lyrics,
            "credits": credits,

            # Friendly aliases used by many custom cards
            "title": getattr(self, "_media_title", None),
            "artist": getattr(self, "_media_artist", None),
            "thumb": getattr(self, "_media_image_url", None),

            # Nice-to-have
            "app_name": "Zing Music",
            "selected_entity_id": getattr(self, "_attr_source", None),
            "selected_label": getattr(self, "_source_label", None),
            "allowed_entity_ids": self._allowed_entity_ids(),
            "subscribed": self._api.is_subscribed, 
        }

    @property
    def media_duration(self) -> Optional[float]:
        return self._media_duration

    # ---------- volume properties ----------

    @property
    def volume_level(self) -> Optional[float]:
        """Return the volume level of the source player."""
        if not self._attr_source:
            return self._volume_level
        
        source_state = self.hass.states.get(self._attr_source)
        if source_state:
            vol = source_state.attributes.get("volume_level")
            if vol is not None:
                self._volume_level = float(vol)
                return self._volume_level
        return self._volume_level

    @property
    def is_volume_muted(self) -> Optional[bool]:
        """Return boolean if volume is currently muted."""
        if not self._attr_source:
            return self._is_volume_muted
        
        source_state = self.hass.states.get(self._attr_source)
        if source_state:
            muted = source_state.attributes.get("is_volume_muted")
            if muted is not None:
                self._is_volume_muted = bool(muted)
                return self._is_volume_muted
        return self._is_volume_muted

    # ---------- repeat / shuffle properties & setters ----------

    @property
    def shuffle(self) -> bool:
        return self._shuffle

    async def async_set_shuffle(self, shuffle: bool) -> None:
        self._shuffle = bool(shuffle)
        # update controller flag and (optionally) reshuffle upcoming items
        self._playback_controller.set_shuffle(self._shuffle)
        _LOGGER.debug(f"[ZING] Shuffle set to {self._shuffle}")
        self.async_write_ha_state()

    @property
    def repeat(self) -> str:
        # HA expects "off" | "one" | "all"
        return self._repeat

    async def async_set_repeat(self, repeat: str) -> None:
        if repeat not in ("off", "one", "all"):
            raise HomeAssistantError(f"Unsupported repeat mode: {repeat}")
        self._repeat = repeat
        # mirror to controller: 'all' -> loop album; 'off'/'one' -> no album loop
        self._playback_controller.set_loop(repeat == "all")
        _LOGGER.debug(f"[ZING] Repeat set to {self._repeat}")
        self.async_write_ha_state()

    # ---------- volume control methods ----------

    async def async_set_volume_level(self, volume: float) -> None:
        """Set volume level on the source player."""
        if not self._attr_source:
            _LOGGER.warning("Cannot set volume: no source selected")
            return
        
        _LOGGER.debug(f"[ZING] Setting volume to {volume} on {self._attr_source}")
        await self.hass.services.async_call(
            "media_player",
            "volume_set",
            {
                "entity_id": self._attr_source,
                "volume_level": volume,
            },
            blocking=True,
        )
        self._volume_level = volume
        self.async_write_ha_state()

    async def async_mute_volume(self, mute: bool) -> None:
        """Mute or unmute the source player."""
        if not self._attr_source:
            _LOGGER.warning("Cannot mute volume: no source selected")
            return
        
        _LOGGER.debug(f"[ZING] Setting mute to {mute} on {self._attr_source}")
        await self.hass.services.async_call(
            "media_player",
            "volume_mute",
            {
                "entity_id": self._attr_source,
                "is_volume_muted": mute,
            },
            blocking=True,
        )
        self._is_volume_muted = mute
        self.async_write_ha_state()

    async def async_volume_up(self) -> None:
        """Turn volume up for the source player."""
        if not self._attr_source:
            _LOGGER.warning("Cannot increase volume: no source selected")
            return
        
        await self.hass.services.async_call(
            "media_player",
            "volume_up",
            {"entity_id": self._attr_source},
            blocking=True,
        )
        # Update local state after a short delay
        await self.async_update()

    async def async_volume_down(self) -> None:
        """Turn volume down for the source player."""
        if not self._attr_source:
            _LOGGER.warning("Cannot decrease volume: no source selected")
            return
        
        _LOGGER.debug(f"[ZING] Decreasing volume on {self._attr_source}")
        await self.hass.services.async_call(
            "media_player",
            "volume_down",
            {"entity_id": self._attr_source},
            blocking=True,
        )
        # Update local state after a short delay
        await self.async_update()

    # ---------- periodic updates ----------

    async def async_update(self):
        # Only mirror duration if we don't have one yet
        if self._media_duration is not None:
            return

        if self._attr_source:
            source_state = self.hass.states.get(self._attr_source)
            if source_state:
                # Update duration
                duration = source_state.attributes.get("media_duration")
                _LOGGER.debug(f"[ZING] async_update: got duration from source: {duration}")
                if duration:
                    try:
                        self._media_duration = float(duration)
                        # _LOGGER.info(f"[ZING] Set media_duration = {self._media_duration}")
                        self.async_write_ha_state()
                    except Exception as e:
                        _LOGGER.warning(f"[ZING] Failed to parse duration in update: {e}")
                
                # Update volume state
                volume_level = source_state.attributes.get("volume_level")
                if volume_level is not None:
                    try:
                        self._volume_level = float(volume_level)
                    except Exception:
                        pass
                
                is_muted = source_state.attributes.get("is_volume_muted")
                if is_muted is not None:
                    self._is_volume_muted = bool(is_muted)

    # ---------- media controls ----------

    async def async_media_seek(self, position: float) -> None:
        self._media_position = position
        self._media_position_updated_at = datetime.now(timezone.utc)

        await self.hass.services.async_call(
            "media_player",
            "media_seek",
            {
                "entity_id": self._attr_source,
                "seek_position": position,
            },
            blocking=True,
        )

        # Try to refresh duration from renderer soon (if it was unknown)
        self._refresh_duration_soon()
        self.async_write_ha_state()
        self._log_now_playing("media_seek")

    async def _play_track(self, track: Track):
        # Record when this track starts to prevent premature auto-advance
        self._track_start_time = time.time()
        
        self._current_track = track
        self._media_url = track.download_url
        
        # Handle image URL with CORS protection
        raw_image_url = (
            (track.album.images.large or track.album.images.medium or track.album.images.small)
            if (track.album and track.album.images) else None
        )
        
        # Proxy external images through Home Assistant to avoid CORS issues
        if raw_image_url and raw_image_url.startswith('http'):
            try:
                # Create a proxied URL using Home Assistant's image proxy
                from urllib.parse import quote
                self._media_image_url = raw_image_url
            except Exception as e:
                _LOGGER.warning(f"Failed to create proxied image URL: {e}")
                self._media_image_url = raw_image_url
        else:
            self._media_image_url = raw_image_url
            
        # Use selected language for display
        self._media_artist = (
            ", ".join(self._get_display_name(a) for a in track.album.artists if self._get_display_name(a))
            if track.album and track.album.artists else "Unknown Artist"
        )
        self._media_title = self._get_display_name(track)
        self._media_album_name = self._get_display_name(track.album) if track.album else "Unknown Album"
        self._state = STATE_PLAYING
        self._media_position = 0.0
        self._media_position_updated_at = datetime.now(timezone.utc)

        # Get media URL first
        if not self._media_url:
            self._media_url = await self.hass.async_add_executor_job(self._api.get_track_download_url, track.id)
            if not self._media_url:
                raise HomeAssistantError(f"No download URL for track {track.id}")

        # Start with API duration immediately for fast playback start
        self._has_reliable_duration = False  # Reset reliable duration flag for new track
        if track.duration:
            try:
                self._media_duration = float(track.duration)
                _LOGGER.debug(f"[ZING] Starting with API duration: {self._media_duration:.2f}s for '{self._get_display_name(track)}'")
            except Exception:
                self._media_duration = None
        else:
            self._media_duration = None

        # Start playback immediately - don't wait for duration detection
        # Use fallback logic for media_content_type to support integrations rejecting 'music'
        await self._play_media_with_fallback(
            media_url=self._media_url,
            title=track.enName,
            artist=self._media_artist,
            thumb=self._media_image_url,
            context="track"
        )

        # Attach listener to renderer state
        self._attach_source_listener()

        # Try to refresh duration from renderer after playback starts (real file duration)
        self._refresh_duration_soon()

        self._log_now_playing("play_track")
        self.async_write_ha_state()

    def _refresh_duration_soon(self):
        """Fetch real duration from renderer and audio file analysis after playback starts."""
        # First pass in ~1 second - check renderer duration
        async def _pull_duration_1(_):
            self._mirror_duration_from_source()
            # Second pass in ~3 seconds to catch late reporters
            async_call_later(self.hass, 3, _pull_duration_2)

        async def _pull_duration_2(_):
            self._mirror_duration_from_source()
            # Duration mirroring complete
            _LOGGER.debug(f"[ZING] Duration mirroring complete: state={self._state}, duration={self._media_duration}")

        # Background audio file duration detection (non-blocking)
        async def _detect_audio_duration(_):
            if self._media_url and self._state == STATE_PLAYING:
                try:
                    # Always detect the actual audio duration to ensure accuracy
                    current_duration = self._media_duration
                    
                    if not current_duration:
                        _LOGGER.debug(f"[ZING] No current duration - will detect from audio file")
                    else:
                        _LOGGER.debug(f"[ZING] Current duration: {current_duration:.2f}s - detecting actual audio file duration")
                    
                    actual_duration = await self._get_actual_audio_duration(self._media_url)
                    if actual_duration:
                        if current_duration and abs(actual_duration - current_duration) > 5:  # Significant difference (lowered threshold)
                            old_duration = self._media_duration
                            self._media_duration = actual_duration
                            self._has_reliable_duration = True  # Mark that we have a reliable duration
                            _LOGGER.info(f"[ZING] Updated to actual audio duration: {actual_duration:.2f}s (was {old_duration:.2f}s)")
                            self.async_write_ha_state()
                        elif not current_duration:
                            # No previous duration, use the detected one
                            self._media_duration = actual_duration
                            self._has_reliable_duration = True
                            _LOGGER.info(f"[ZING] Set actual audio duration: {actual_duration:.2f}s")
                            self.async_write_ha_state()
                        else:
                            _LOGGER.debug(f"[ZING] Audio file duration ({actual_duration:.2f}s) similar to current ({current_duration:.2f}s) - keeping current")
                    else:
                        _LOGGER.debug(f"[ZING] Could not detect actual audio duration")
                            
                except Exception as e:
                    _LOGGER.debug(f"[ZING] Background duration detection failed: {e}")

        async_call_later(self.hass, 1, _pull_duration_1)
        # Enable smart background duration detection - only when API duration seems wrong
        async_call_later(self.hass, 2, _detect_audio_duration)

    def _mirror_duration_from_source(self):
        # Don't override duration if we have a reliable audio file duration
        if hasattr(self, '_has_reliable_duration') and self._has_reliable_duration:
            _LOGGER.debug(f"[ZING] Skipping renderer duration override - using reliable audio file duration: {self._media_duration}")
        else:
            if not self._attr_source:
                return
            source_state = self.hass.states.get(self._attr_source)
            if not source_state:
                return
            duration = source_state.attributes.get("media_duration")
            if duration:
                try:
                    new_val = float(duration)
                    if self._media_duration != new_val:
                        _LOGGER.debug(f"[ZING] Overwriting media_duration from renderer: {new_val}")
                        self._media_duration = new_val
                except Exception:
                    pass
        
        # Always mirror position from renderer for accuracy
        if self._attr_source:
            source_state = self.hass.states.get(self._attr_source)
            if source_state:
                # Mirror position from renderer
                renderer_position = source_state.attributes.get("media_position")
                if renderer_position is not None:
                    try:
                        new_position = float(renderer_position)
                        current_calculated_position = self.media_position  # This includes elapsed time calculation
                        
                        # Debug logging to understand position sync
                        _LOGGER.debug(f"[ZING] Mirror check - Renderer: {new_position:.1f}s, calculated: {current_calculated_position:.1f}s")
                        
                        # Update more frequently to keep UI in sync (lower threshold)
                        if abs(current_calculated_position - new_position) > 0.5:  # 0.5 second threshold
                            _LOGGER.info(f"[ZING] Mirror syncing position: {new_position:.1f}s (was calculated as {current_calculated_position:.1f}s)")
                            self._media_position = new_position
                            self._media_position_updated_at = datetime.now(timezone.utc)
                            # Notify Home Assistant of position change
                            self.async_write_ha_state()
                    except Exception as e:
                        _LOGGER.debug(f"[ZING] Failed to mirror position: {e}")
                        pass
    def _check_subscription(self):
        """Raise error if user is not subscribed."""
        if not self._api.is_subscribed:
            raise HomeAssistantError("Zing Music subscription required. Please subscribe to use this feature.")

    async def async_play_media(self, media_type: str, media_id: str, **kwargs) -> None:
        _LOGGER.debug(f"Attempting to play media: type={media_type}, id={media_id}")
        self._check_subscription()
        try:
            # Resume / Seek URL
            if media_type in ("music", MediaType.MUSIC) and isinstance(media_id, str) and media_id.startswith("http"):
                self._media_url = media_id
                self._state = STATE_PLAYING
                self._media_position_updated_at = datetime.now(timezone.utc)

                _LOGGER.debug("Replaying known media URL (seek or resume)")

                await self._play_media_with_fallback(
                    media_url=media_id,
                    title=self._media_title,
                    artist=self._media_artist,
                    thumb=self._media_image_url,
                    context="resume"
                )

                self._attach_source_listener()
                self._refresh_duration_soon()
                self.async_write_ha_state()
                self._log_now_playing("resume_url")
                return

            # Album track with explicit context "album_id:track_id"
            if media_type == "album_track":
                album_id_str, track_id_str = media_id.split(":")
                album_id = int(album_id_str)
                track_id = int(track_id_str)

                _LOGGER.debug(f"[ZING] Playing album track: album={album_id}, track={track_id}")

                # honor shuffle flag when building the playlist
                track = await self.hass.async_add_executor_job(
                    self._playback_controller.play_album, album_id, track_id, self._shuffle
                )
                if not track or not track.download_url:
                    raise HomeAssistantError(f"Invalid album track: {track_id}")
                await self._play_track(track)
                return

            # Album Playback
            if media_type in ("album", MediaType.ALBUM):
                album_id = int(media_id)
                track = await self.hass.async_add_executor_job(
                    self._playback_controller.play_album, album_id, None, self._shuffle
                )
                if track:
                    await self._play_track(track)
                self._state = STATE_PLAYING
                self._media_title = f"Album #{album_id}"
                self._media_artist = ""
                self._media_album_name = ""
                self._media_image_url = None
                self._media_position = 0
                self._media_position_updated_at = datetime.now(timezone.utc)
                self.async_write_ha_state()
                return

            # Track Playback (no explicit album context)
            if media_type in ("track", MediaType.TRACK):
                # Check if this is a playlist track (format: "playlist:playlist_id:track_id")
                if media_id.startswith("playlist:"):
                    parts = media_id.split(":")
                    if len(parts) == 3:
                        playlist_id = int(parts[1])
                        track_id = int(parts[2])
                        
                        _LOGGER.debug(f"[ZING] Playing track {track_id} from playlist {playlist_id}")
                        
                        # Load the entire playlist and start from the selected track
                        track = await self.hass.async_add_executor_job(
                            self._playback_controller.play_playlist, playlist_id, track_id, self._shuffle
                        )
                        if not track:
                            raise HomeAssistantError(f"Track {track_id} not found in playlist {playlist_id}")
                        
                        await self._play_track(track)
                        return
                
                # Regular track playback
                track_id = int(media_id)

                # Try to find the track in the current playlist first
                track = next(
                    (t for t in self._playback_controller.playlist if t.id == track_id),
                    None
                )

                if not track:
                    # Fallback to fetch from API
                    track = await self.hass.async_add_executor_job(self._api.get_track_by_id, track_id)
                    if not track:
                        raise HomeAssistantError(f"Track {track_id} not found")

                    # If album is missing, cannot set playlist context → standalone
                    if not track.album:
                        _LOGGER.warning(f"[ZING] Track {track_id} has no album — playing in standalone mode.")
                        await self._play_track(track)
                        return

                    # Rebuild playlist from album with this track selected (respect shuffle)
                    await self.hass.async_add_executor_job(
                        self._playback_controller.play_album, track.album.id, track.id, self._shuffle
                    )
                    track = self._playback_controller.get_current_track()
                else:
                    # Track is already part of current playlist — set index
                    for i, t in enumerate(self._playback_controller.playlist):
                        if t.id == track.id:
                            self._playback_controller.current_index = i
                            break
                    # Update track reference to the one from the playlist at the correct index
                    track = self._playback_controller.get_current_track()

                if not track.download_url:
                    raise HomeAssistantError(f"Track {track.id} has no download URL")

                await self._play_track(track)
                return

            # Playlist Playback
            if media_type in ("playlist", MEDIA_TYPE_PLAYLIST, MEDIA_TYPE_FEATURED_PLAYLIST, MEDIA_TYPE_ARTISTS_PLAYLIST):
                playlist_id = int(media_id)
                _LOGGER.debug(f"[ZING] Playing playlist: {playlist_id}")

                # Use the playback controller to load playlist into queue
                track = await self.hass.async_add_executor_job(
                    self._playback_controller.play_playlist, playlist_id, None, self._shuffle
                )
                if not track:
                    raise HomeAssistantError("Playlist is empty or failed to load")
                
                await self._play_track(track)
                return

            raise HomeAssistantError(f"Unsupported media type: {media_type}")

        except Exception as e:
            _LOGGER.error(f"Error playing media: {e}")
            self._state = STATE_IDLE
            self.async_write_ha_state()
            raise HomeAssistantError(f"Failed to play media: {e}")

    # ----- Group Speaker Support Methods -----

    def _is_speaker_group(self, entity_id: str) -> bool:
        """Check if the entity is a speaker group."""
        if not entity_id:
            return False
        
        entity_state = self.hass.states.get(entity_id)
        if not entity_state:
            return False
        
        
        # Check for group indicators
        domain = entity_id.split(".")[0]
        if domain == "group":
            return True
        
        # Check for group attributes - even single-member groups are still groups!
        group_members = entity_state.attributes.get("entity_id", [])
        if group_members and len(group_members) >= 1:  # Changed from > 1 to >= 1
            return True
        
        # Check for Sonos groups
        if "sonos" in entity_id.lower() and entity_state.attributes.get("group_members"):
            return True
        
        # Check for Cast groups  
        if entity_state.attributes.get("app_name") == "Spotify" and entity_state.attributes.get("group_members"):
            return True
        
        return False

    async def _play_to_group_members(self, group_entity_id: str, media_url: str, title: str, artist: str, thumb: str):
        """Play media to all members of a speaker group."""
        try:
            # Get the group entity state
            group_state = self.hass.states.get(group_entity_id)
            if not group_state:
                _LOGGER.error(f"[ZING] Group entity {group_entity_id} not found")
                return False
            
            _LOGGER.info(f"[ZING] Group entity {group_entity_id} state: {group_state.state}")
            _LOGGER.info(f"[ZING] Group entity attributes: {group_state.attributes}")
            
            # Check if this is a group entity
            group_members = group_state.attributes.get("entity_id", [])
            if not group_members:
                # Not a group or no members, fallback to single entity
                _LOGGER.warning(f"[ZING] No 'entity_id' attribute found for group {group_entity_id}")
                return False
            
            _LOGGER.info(f"[ZING] Playing to group {group_entity_id} with {len(group_members)} members: {group_members}")
            
            # Log details about each member before playing
            for member_entity in group_members:
                member_state = self.hass.states.get(member_entity)
                if member_state:
                    _LOGGER.info(f"[ZING] Group member {member_entity}: state={member_state.state}, platform={member_state.attributes.get('source', 'unknown')}")
                else:
                    _LOGGER.warning(f"[ZING] Group member {member_entity} not found in states")
            
            # Play to each group member individually
            success_count = 0
            failed_members = []
            
            for member_entity in group_members:
                try:
                    _LOGGER.info(f"[ZING] Attempting playback to group member: {member_entity}")
                    await self._play_single_entity(
                        entity_id=member_entity,
                        media_url=media_url,
                        title=title,
                        artist=artist, 
                        thumb=thumb,
                        context=f"group_member_{member_entity}"
                    )
                    success_count += 1
                    _LOGGER.info(f"[ZING] ✅ Successfully started playback on group member: {member_entity}")
                except Exception as e:
                    failed_members.append(f"{member_entity}: {str(e)}")
                    _LOGGER.error(f"[ZING] ❌ Failed to play to group member {member_entity}: {e}")
            
            if success_count > 0:
                _LOGGER.info(f"[ZING] Group playback completed: {success_count}/{len(group_members)} speakers successful")
                if failed_members:
                    _LOGGER.warning(f"[ZING] Failed members: {failed_members}")
                return True
            else:
                _LOGGER.error(f"[ZING] Failed to start playback on any group members. All failures: {failed_members}")
                return False
                
        except Exception as e:
            _LOGGER.error(f"[ZING] Error handling group playback: {e}")
            return False

    async def _play_single_entity(self, entity_id: str, media_url: str, title: str | None, artist: str | None, thumb: str | None, context: str):
        """Play media to a single entity with professional Cast device support."""
        # Detect platform hints from entity_id / friendly name
        platform_hint = ""
        if any(pat in entity_id for pat in ("androidtv", "androidtv_remote")):
            platform_hint = "androidtv"
        elif any(pat in entity_id for pat in ("chromecast", "cast")):
            platform_hint = "cast"
        else:
            st = self.hass.states.get(entity_id)
            if st:
                fname = (st.attributes.get("friendly_name") or "").lower()
                if "android" in fname:
                    platform_hint = "androidtv"
                elif "cast" in fname or "google" in fname or "nest" in fname:
                    platform_hint = "cast"

        _LOGGER.debug(f"[ZING] Playing to {entity_id} (platform_hint={platform_hint}, context={context})")

        # Professional Cast Integration Approach (like official Home Assistant Cast integration)
        if platform_hint == "cast":
            return await self._play_cast_device(entity_id, media_url, title, artist, thumb, context)
        
        # Standard media player approach for non-Cast devices
        if platform_hint == "androidtv":
            candidates = ["url", "music", "audio"]
        else:
            candidates = ["music", "url", "audio"]

        base = {
            "entity_id": entity_id,
            "media_content_id": media_url,
        }
        extra = {"title": title, "artist": artist, "thumb": thumb}
        # Remove empty values from extra
        extra = {k: v for k, v in extra.items() if v}
        if extra:
            base["extra"] = extra

        _LOGGER.debug(f"[ZING] Will try media types {candidates} for {entity_id} with extra={bool(extra)}")

        last_exc: Exception | None = None
        for mtype in candidates:
            # Try with metadata first
            service_data = {**base, "media_content_type": mtype}
            try:
                _LOGGER.debug(f"[ZING] Attempting play_media: type='{mtype}' entity={entity_id} with_extra={bool('extra' in service_data)}")
                await self.hass.services.async_call(
                    "media_player",
                    "play_media",
                    service_data,
                    blocking=True,
                )
                _LOGGER.info(f"[ZING] ✅ play_media success: type='{mtype}' context={context} entity={entity_id}")
                return
            except ValueError as exc:  # Invalid media type
                if "Invalid media type" in str(exc):
                    _LOGGER.debug(f"[ZING] media type '{mtype}' rejected for {entity_id}: {exc}; trying next")
                    last_exc = exc
                    continue
                last_exc = exc
                _LOGGER.warning(f"[ZING] ValueError during play_media type='{mtype}' (not retrying): {exc}")
                break
            except TypeError as exc:  # Unexpected keyword argument
                if "unexpected keyword argument" in str(exc) and "extra" in service_data:
                    _LOGGER.debug(f"[ZING] Extra metadata rejected for {entity_id}: {exc}; retrying without extra")
                    # Retry same media type without extra metadata
                    try:
                        minimal_data = {
                            "entity_id": entity_id,
                            "media_content_id": media_url,
                            "media_content_type": mtype
                        }
                        await self.hass.services.async_call(
                            "media_player",
                            "play_media",
                            minimal_data,
                            blocking=True,
                        )
                        _LOGGER.info(f"[ZING] ✅ play_media success: type='{mtype}' (no extra) context={context} entity={entity_id}")
                        return
                    except Exception as retry_exc:  # noqa: BLE001
                        _LOGGER.debug(f"[ZING] Retry without extra also failed for type='{mtype}': {retry_exc}")
                        last_exc = retry_exc
                        continue
                else:
                    last_exc = exc
                    _LOGGER.warning(f"[ZING] TypeError during play_media type='{mtype}': {exc}; trying next")
                    continue
            except Exception as exc:  # noqa: BLE001
                last_exc = exc
                _LOGGER.error(f"[ZING] ❌ Unexpected error play_media type='{mtype}' context={context} entity={entity_id}: {exc}; trying next")
                continue

        _LOGGER.error(f"[ZING] ❌ Failed to play media to {entity_id} after trying {candidates} (last error: {last_exc})")
        raise HomeAssistantError(f"Failed to play media to {entity_id} after trying {candidates} (last error: {last_exc})")

    async def _play_cast_device(self, entity_id: str, media_url: str, title: str | None, artist: str | None, thumb: str | None, context: str):
        """Play media to Cast device using professional Cast protocol (like official HA Cast integration)."""
        _LOGGER.info(f"[ZING] Using professional Cast playback for {entity_id}")
        
        # Build metadata in Cast format (same as official Cast integration)
        # The Cast integration expects specific metadata structure
        metadata = {}
        if title:
            metadata["title"] = title
        # Note: 'artist' is not a direct field in Cast metadata, it goes under different keys
        # The official Cast integration handles this differently
        if thumb:
            metadata["images"] = [{"url": thumb}]
        
        # Use Cast-optimized media type (audio for music content, like official integration)
        media_type = "audio"
        
        # Build service call data using Cast integration pattern
        # Don't pass artist as direct metadata to avoid the TypeError
        service_data = {
            "entity_id": entity_id,
            "media_content_type": media_type,
            "media_content_id": media_url,
        }
        
        # Add metadata in extra field if available (Cast integration pattern)
        # Only include supported metadata fields
        if metadata:
            service_data["extra"] = metadata
        
        _LOGGER.debug(f"[ZING] Cast service call: {service_data}")
        
        try:
            await self.hass.services.async_call(
                "media_player", "play_media", service_data, blocking=True
            )
            _LOGGER.info(f"[ZING] ✅ Successfully played to Cast device {entity_id}")
        except Exception as exc:
            _LOGGER.error(f"[ZING] ❌ Failed to play to Cast device {entity_id}: {exc}")
            # Fallback: try with minimal metadata if full metadata fails
            _LOGGER.warning(f"[ZING] Attempting minimal metadata fallback for Cast device {entity_id}")
            
            minimal_service_data = {
                "entity_id": entity_id,
                "media_content_type": media_type,
                "media_content_id": media_url,
            }
            
            # Try just title if that was the issue
            if title:
                minimal_service_data["extra"] = {"title": title}
            
            try:
                await self.hass.services.async_call(
                    "media_player", "play_media", minimal_service_data, blocking=True
                )
                _LOGGER.info(f"[ZING] ✅ Cast minimal metadata fallback successful for {entity_id}")
                return
            except Exception as minimal_exc:
                _LOGGER.debug(f"[ZING] Cast minimal metadata failed: {minimal_exc}")
            
            # Final fallback: try with standard media types if Cast-specific approach fails
            _LOGGER.warning(f"[ZING] Attempting standard media type fallback for Cast device {entity_id}")
            candidates = ["url", "music", "audio"]
            
            base = {
                "entity_id": entity_id,
                "media_content_id": media_url,
            }
            # Don't include extra metadata in fallback to avoid more errors
            
            last_exc = exc
            for mtype in candidates:
                service_data = {**base, "media_content_type": mtype}
                try:
                    _LOGGER.debug(f"[ZING] Cast fallback: trying {mtype} for {entity_id}")
                    await self.hass.services.async_call(
                        "media_player", "play_media", service_data, blocking=True
                    )
                    _LOGGER.info(f"[ZING] ✅ Cast fallback successful: {mtype} for {entity_id}")
                    return
                except Exception as fallback_exc:  # noqa: BLE001
                    _LOGGER.debug(f"[ZING] Cast fallback {mtype} failed: {fallback_exc}")
                    last_exc = fallback_exc
                    continue
            
            raise HomeAssistantError(f"Failed to play to Cast device {entity_id}: {last_exc}")

    # ----- internal helpers for play_media fallbacks -----

    async def _play_media_with_fallback(self, media_url: str, title: str | None, artist: str | None, thumb: str | None, context: str):
        """Play media with group support and fallback handling."""
        if not self._attr_source:
            raise HomeAssistantError("No target source player selected")

        entity_id = self._attr_source

        # Debug group detection
        is_group = self._is_speaker_group(entity_id)
        _LOGGER.info(f"[ZING] _play_media_with_fallback: entity_id={entity_id}, is_group={is_group}")

        # Check if this is a speaker group
        if is_group:
            _LOGGER.info(f"[ZING] Detected speaker group: {entity_id} - playing to individual members")
            
            # Skip group-level playback entirely - go straight to individual members
            # This is the most reliable approach as you suggested
            group_success = await self._play_to_group_members(entity_id, media_url, title, artist, thumb)
            if group_success:
                self._attr_media_content_type = "music"  # Set a default
                return
            else:
                raise HomeAssistantError(f"Failed to play to any members of group {entity_id}")
        else:
            # Single entity playback (original logic)
            await self._play_single_entity(entity_id, media_url, title, artist, thumb, context)
            self._attr_media_content_type = "music"  # Set a default

    async def async_browse_media(self, media_content_type: Optional[str] = None, media_content_id: Optional[str] = None) -> BrowseMedia:

        # Special: "Playing" tab / current queue
        if (
            media_content_type == "playing"
            or media_content_id in ("playing", "current")
        ):
            children = []
            playlist = getattr(self._playback_controller, "playlist", []) or []
            cur_index = getattr(self._playback_controller, "current_index", 0) or 0

            for idx, t in enumerate(playlist):
                # Build a playable item for each track in the current queue
                has_album = getattr(t, "album", None) is not None
                content_id = f"{t.album.id}:{t.id}" if has_album else str(t.id)
                content_type = "album_track" if has_album else MediaType.TRACK
                thumb = (
                    (t.album.images.small or t.album.images.medium or t.album.images.large)
                    if (has_album and getattr(t.album, "images", None)) else None
                )
                title = getattr(t, "enName", None) or getattr(t, "heName", None) or f"Track {t.id}"

                children.append(
                    BrowseMedia(
                        media_class="track",
                        media_content_id=content_id,
                        media_content_type=content_type,
                        title=("▶ " + title) if idx == cur_index else title,
                        can_play=True,
                        can_expand=False,
                        thumbnail=thumb,
                    )
                )

            return BrowseMedia(
                media_class="playlist",
                media_content_id="playing",
                media_content_type="playing",
                title="Playing",
                can_play=False,
                can_expand=True,
                children=children,
                children_media_class="track",
            )

        if media_content_type == "search_results" and self._search_result:
            if media_content_id == self._search_result.media_content_id:
                return self._search_result

        if media_content_id is None:
            children = []

            if self._search_result:
                children.append(self._search_result)

            children.append(
                BrowseMedia(
                    media_class="playlist",
                    media_content_id="playing",
                    media_content_type="playing",
                    title="Playing",
                    can_play=False,
                    can_expand=True,
                    children_media_class="track",
                )
            )

            # Check if kids-only mode is enabled
            kids_only = self._is_kids_only_mode()
            
            if kids_only:
                # In kids-only mode, show only kids content
                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_KIDS_ARTISTS,
                        media_content_type=MEDIA_TYPE_KIDS_ARTISTS,
                        title="Kids",
                        can_play=False,
                        can_expand=True,
                    )
                )
                
                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_KIDS,
                        media_content_type=MEDIA_TYPE_KIDS,
                        title="Latest Kids",
                        can_play=False,
                        can_expand=True,
                    )
                )
            else:
                # Normal mode - show all content types
                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_ARTIST,
                        media_content_type=MEDIA_TYPE_ARTIST,
                        title="Artists",
                        can_play=False,
                        can_expand=True,
                    )
                )

                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_LATEST,
                        media_content_type=MEDIA_TYPE_LATEST,
                        title="Latest",
                        can_play=False,
                        can_expand=True,
                    )
                )

                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_FEATURED_PLAYLIST,
                        media_content_type=MEDIA_TYPE_FEATURED_PLAYLIST,
                        title="Featured Playlists",
                        can_play=False,
                        can_expand=True,
                    )
                )

                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_ARTISTS_PLAYLIST,
                        media_content_type=MEDIA_TYPE_ARTISTS_PLAYLIST,
                        title="Artists Playlists",
                        can_play=False,
                        can_expand=True,
                    )
                )

                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_PODCAST_HOSTS,
                        media_content_type=MEDIA_TYPE_PODCAST_HOSTS,
                        title="Podcasts",
                        can_play=False,
                        can_expand=True,
                    )
                )

                children.append(
                    BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_KIDS_ARTISTS,
                        media_content_type=MEDIA_TYPE_KIDS_ARTISTS,
                        title="Kids",
                        can_play=False,
                        can_expand=True,
                    )
                )

            return BrowseMedia(
                media_class="directory",
                media_content_id="root",
                media_content_type="root",
                title="Zing Music",
                can_play=False,
                can_expand=True,
                children=children,
            )

        if media_content_type == MEDIA_TYPE_ARTIST:
            if media_content_id and media_content_id.isdigit():
                try:
                    artist_id = int(media_content_id)
                    _artist, albums, featured_albums = await self.hass.async_add_executor_job(self._api.get_artist_albums, artist_id)

                    # Show artist subfolders instead of direct albums
                    children = []
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_ARTIST_LATEST}:{artist_id}",
                            media_content_type=MEDIA_TYPE_ARTIST_LATEST,
                            title="Latest Releases",
                            can_play=False,
                            can_expand=True,
                        )
                    )
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_ARTIST_ALBUMS}:{artist_id}",
                            media_content_type=MEDIA_TYPE_ARTIST_ALBUMS,
                            title="Albums",
                            can_play=False,
                            can_expand=True,
                        )
                    )
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_ARTIST_SINGLES}:{artist_id}",
                            media_content_type=MEDIA_TYPE_ARTIST_SINGLES,
                            title="Singles",
                            can_play=False,
                            can_expand=True,
                        )
                    )
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_ARTIST_SONGS}:{artist_id}",
                            media_content_type=MEDIA_TYPE_ARTIST_SONGS,
                            title="Songs",
                            can_play=False,
                            can_expand=True,
                        )
                    )
                    
                    # Add "Related" node only if there are featured albums
                    if featured_albums:
                        children.append(
                            BrowseMedia(
                                media_class="directory",
                                media_content_id=f"{MEDIA_TYPE_ARTIST_RELATED}:{artist_id}",
                                media_content_type=MEDIA_TYPE_ARTIST_RELATED,
                                title="Related",
                                can_play=False,
                                can_expand=True,
                            )
                        )

                    return BrowseMedia(
                        media_class="artist",
                        media_content_id=media_content_id,
                        media_content_type=MEDIA_TYPE_ARTIST,
                        title=self._get_display_name(_artist),
                        can_play=False,
                        can_expand=True,
                        children=children,
                    )
                except Exception as e:
                    _LOGGER.error(f"Error Browse artist {media_content_id}: {e}")
                    raise HomeAssistantError(f"Error Browse artist: {e}")

            elif media_content_id == MEDIA_TYPE_ARTIST:
                try:
                    all_artists = await self.hass.async_add_executor_job(self._api.get_all_artists)
                    
                    # Use selected language for display
                    children = []
                    for artist in all_artists:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(artist):
                            continue
                        children.append(
                            BrowseMedia(
                                media_class="artist",
                                media_content_id=str(artist.id),
                                media_content_type=MEDIA_TYPE_ARTIST,
                                title=self._get_display_name(artist),
                                can_play=False,
                                can_expand=True,
                                thumbnail=(
                                    artist.images.small
                                    or artist.images.medium
                                    or artist.images.large
                                    if artist.images else None
                                ),
                            )
                        )
                    return BrowseMedia(
                        media_class="directory",
                        media_content_id=MEDIA_TYPE_ARTIST,
                        media_content_type=MEDIA_TYPE_ARTIST,
                        title="All Artists",
                        can_play=False,
                        can_expand=True,
                        children=children,
                    )
                except Exception as e:
                    _LOGGER.error(f"Error loading all artists: {e}")
                    raise HomeAssistantError(f"Error loading artists: {e}")

        if media_content_type == MEDIA_TYPE_ALBUM:
            try:
                album_id = int(media_content_id)
                _LOGGER.debug(f"Fetching album data for album ID {album_id}")
                album_data = await self.hass.async_add_executor_job(self._api.get_album_tracks, album_id)
                
                # Check if album_data is valid
                if not album_data:
                    _LOGGER.error(f"No album data returned for album ID {album_id}")
                    raise HomeAssistantError(f"Album {album_id} not found or unavailable")
                
                _LOGGER.debug(f"Album data type: {type(album_data)}, has tracks: {hasattr(album_data, 'tracks')}")
                
                # Check if tracks exist
                tracks = getattr(album_data, 'tracks', None)
                if tracks is None:
                    _LOGGER.error(f"No tracks found for album ID {album_id}")
                    tracks = []
                
                _LOGGER.debug(f"Found {len(tracks)} tracks for album {album_id}")
                
                # Use selected language for display
                children = []
                for i, track in enumerate(tracks):
                    if track:  # Additional null check for each track
                        try:
                            children.append(
                                BrowseMedia(
                                    media_class="track",
                                    media_content_id=f"{album_data.id}:{track.id}",
                                    media_content_type="album_track",
                                    title=self._get_display_name(track),
                                    can_play=True,
                                    can_expand=False,
                                    thumbnail=album_data.images.small if hasattr(album_data, 'images') and album_data.images else None,
                                )
                            )
                        except Exception as track_error:
                            _LOGGER.error(f"Error processing track {i}: {track_error}")
                
                return BrowseMedia(
                    media_class="album",
                    media_content_id=media_content_id,
                    media_content_type=MEDIA_TYPE_ALBUM,
                    title=self._get_display_name(album_data),
                    can_play=False,
                    can_expand=True,
                    children=children,
                )
            except Exception as e:
                _LOGGER.error(f"Error Browse album {media_content_id}: {e}")
                import traceback
                _LOGGER.error(f"Traceback: {traceback.format_exc()}")
                raise HomeAssistantError(f"Error Browse album: {e}")

        if media_content_type == MediaType.TRACK:
            _LOGGER.warning(f"Attempted to browse a track: {media_content_id}. Tracks are not expandable.")
            return BrowseMedia(
                media_class="track",
                media_content_id=media_content_id,
                media_content_type=MediaType.TRACK,
                title="Track",
                can_play=True,
                can_expand=False,
                children=[],
            )

        elif media_content_type == MEDIA_TYPE_RECENT_TRACKS:
            # Parse limit (default 100)
            try:
                limit = 100
                if media_content_id and ":" in media_content_id:
                    _, lim = media_content_id.split(":", 1)
                    limit = max(1, min(200, int(lim or 100)))  # cap to 200 for safety
            except Exception:
                limit = 100

            tracks = await self.hass.async_add_executor_job(self._api.get_recent_tracks, limit)

            # Use selected language for display
            children = []
            for t in tracks:
                # Filter based on whitelist and blacklist
                if not self._is_artist_allowed(t):
                    continue
                    
                # Pick the best available image
                thumb = None
                if getattr(t, "album", None) and getattr(t.album, "images", None):
                    imgs = t.album.images
                    thumb = getattr(imgs, "small", None) or getattr(imgs, "medium", None) or getattr(imgs, "large", None)

                title = self._get_display_name(t) or f"Track {t.id}"
                artist = None
                if getattr(t, "album", None) and getattr(t.album, "artists", None):
                    # join distinct artist names if multiple
                    names = []
                    for a in (t.album.artists or []):
                        nm = self._get_display_name(a)
                        if nm:
                            names.append(nm)
                    artist = ", ".join(names) if names else None

                children.append(
                    BrowseMedia(
                        title=title,
                        media_class="music",
                        media_content_type=MEDIA_TYPE_TRACK,
                        media_content_id=str(t.id),     # your play handler should accept this id
                        can_play=True,
                        can_expand=False,
                        thumbnail=thumb,
                    )
                )

            return BrowseMedia(
                title=f"Last {limit} songs",
                media_class="directory",
                media_content_type=MEDIA_TYPE_RECENT_TRACKS,
                media_content_id=f"recent:{limit}",
                can_play=False,
                can_expand=True,
                children=children,
                children_media_class="music",
            )

        elif media_content_type == MEDIA_TYPE_LATEST:
            # Show Latest submenu with Albums, Singles, Songs, Kids, Podcasts
            children = []
            
            children.append(
                BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_LATEST_ALBUMS,
                    media_content_type=MEDIA_TYPE_LATEST_ALBUMS,
                    title="Albums",
                    can_play=False,
                    can_expand=True,
                )
            )
            
            children.append(
                BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_LATEST_SINGLES,
                    media_content_type=MEDIA_TYPE_LATEST_SINGLES,
                    title="Singles",
                    can_play=False,
                    can_expand=True,
                )
            )
            
            children.append(
                BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_LATEST_SONGS,
                    media_content_type=MEDIA_TYPE_LATEST_SONGS,
                    title="Songs",
                    can_play=False,
                    can_expand=True,
                )
            )
            
            children.append(
                BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_KIDS,
                    media_content_type=MEDIA_TYPE_KIDS,
                    title="Kids",
                    can_play=False,
                    can_expand=True,
                )
            )
            
            children.append(
                BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_PODCAST,
                    media_content_type=MEDIA_TYPE_PODCAST,
                    title="Podcasts",
                    can_play=False,
                    can_expand=True,
                )
            )
            
            return BrowseMedia(
                media_class="directory",
                media_content_type=MEDIA_TYPE_LATEST,
                media_content_id=MEDIA_TYPE_LATEST,
                title="Latest",
                can_play=False,
                can_expand=True,
                children=children,
            )

        elif media_content_type == MEDIA_TYPE_LATEST_ALBUMS:
            # Get latest albums
            try:
                # Parse limit (default from config)
                limit = self._get_latest_count()
                if media_content_id and ":" in media_content_id:
                    _, lim = media_content_id.split(":", 1)
                    limit = max(1, min(200, int(lim or limit)))
            except Exception:
                limit = self._get_latest_count()
            
            albums = await self.hass.async_add_executor_job(self._api.get_latest_albums, limit)
            children = []
            
            for album in albums:
                if not self._is_artist_allowed(album):
                    continue
                    
                # Get artist names
                artist_names = []
                if album.artists:
                    artist_names.extend(self._get_display_name(a) for a in album.artists if self._get_display_name(a))
                if album.featuredArtists:
                    artist_names.extend(self._get_display_name(a) for a in album.featuredArtists if self._get_display_name(a))
                artist_str = ", ".join(artist_names) if artist_names else "Unknown Artist"
                
                children.append(
                    BrowseMedia(
                        media_class="album",
                        media_content_id=str(album.id),
                        media_content_type=MEDIA_TYPE_ALBUM,
                        title=f"{self._get_display_name(album)} - {artist_str}",
                        can_play=False,
                        can_expand=True,
                        thumbnail=album.images.small if album.images else None,
                    )
                )
            
            return BrowseMedia(
                media_class="directory",
                media_content_type=MEDIA_TYPE_LATEST_ALBUMS,
                media_content_id=f"{MEDIA_TYPE_LATEST_ALBUMS}:{limit}",
                title="Latest Albums",
                can_play=False,
                can_expand=True,
                children=children,
            )

        elif media_content_type == MEDIA_TYPE_LATEST_SINGLES:
            # Get latest singles
            try:
                # Parse limit (default from config)
                limit = self._get_latest_count()
                if media_content_id and ":" in media_content_id:
                    _, lim = media_content_id.split(":", 1)
                    limit = max(1, min(200, int(lim or limit)))
            except Exception:
                limit = self._get_latest_count()
            
            singles = await self.hass.async_add_executor_job(self._api.get_latest_singles, limit)
            children = []
            
            for single in singles:
                if not self._is_artist_allowed(single):
                    continue
                    
                # Get artist names from the track's album
                artist_names = []
                if single.album and single.album.artists:
                    artist_names.extend(self._get_display_name(a) for a in single.album.artists if self._get_display_name(a))
                if single.album and single.album.featuredArtists:
                    artist_names.extend(self._get_display_name(a) for a in single.album.featuredArtists if self._get_display_name(a))
                artist_str = ", ".join(artist_names) if artist_names else "Unknown Artist"
                
                children.append(
                    BrowseMedia(
                        media_class="album",
                        media_content_id=str(single.album.id) if single.album else str(single.id),
                        media_content_type=MEDIA_TYPE_ALBUM,
                        title=f"{self._get_display_name(single)} - {artist_str}",
                        can_play=False,
                        can_expand=True,
                        thumbnail=single.album.images.small if single.album and single.album.images else None,
                    )
                )
            
            return BrowseMedia(
                media_class="directory",
                media_content_type=MEDIA_TYPE_LATEST_SINGLES,
                media_content_id=f"{MEDIA_TYPE_LATEST_SINGLES}:{limit}",
                title="Latest Singles",
                can_play=False,
                can_expand=True,
                children=children,
            )

        elif media_content_type == MEDIA_TYPE_LATEST_SONGS:
            # Get latest songs
            try:
                # Parse limit (default from config)
                limit = self._get_latest_count()
                if media_content_id and ":" in media_content_id:
                    _, lim = media_content_id.split(":", 1)
                    limit = max(1, min(200, int(lim or limit)))
            except Exception:
                limit = self._get_latest_count()
            
            tracks = await self.hass.async_add_executor_job(self._api.get_latest_songs)
            children = []
            
            for track in tracks:
                if not self._is_artist_allowed(track):
                    continue
                    
                # Get artist name from album if available
                artist_name = "Unknown Artist"
                if track.album and track.album.artists:
                    artist_name = ", ".join(self._get_display_name(a) for a in track.album.artists if self._get_display_name(a))
                
                # Get album image if available
                thumbnail = None
                if track.album and track.album.images:
                    thumbnail = track.album.images.small or track.album.images.medium or track.album.images.large
                
                # Build content ID for track (album:track for album context, or just track)
                has_album = getattr(track, "album", None) is not None
                content_id = f"{track.album.id}:{track.id}" if has_album else str(track.id)
                content_type = "album_track" if has_album else MediaType.TRACK
                
                children.append(
                    BrowseMedia(
                        media_class="track",
                        media_content_id=content_id,
                        media_content_type=content_type,
                        title=f"{self._get_display_name(track)} - {artist_name}",
                        can_play=True,
                        can_expand=False,
                        thumbnail=thumbnail,
                    )
                )
            
            return BrowseMedia(
                media_class="directory",
                media_content_type=MEDIA_TYPE_LATEST_SONGS,
                media_content_id=f"{MEDIA_TYPE_LATEST_SONGS}:{limit}",
                title="Latest Songs",
                can_play=False,
                can_expand=True,
                children=children,
                children_media_class="track",
            )

        elif media_content_type == MEDIA_TYPE_ARTIST_LATEST:
            # Artist Latest Releases
            if ":" in media_content_id:
                artist_id = int(media_content_id.split(":")[1])
                latest_releases_data, _ = await self.hass.async_add_executor_job(self._api.get_artist_latest_releases, artist_id)
                
                children = []
                for item_data in latest_releases_data:
                    # Parse raw data to Album object
                    if isinstance(item_data, dict):
                        item = self._api._parse_album(item_data)
                    else:
                        item = item_data  # Already parsed
                        
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(item.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(item),
                            can_play=False,
                            can_expand=True,
                            thumbnail=item.images.small if item.images else None,
                        )
                    )
                
                return BrowseMedia(
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_ARTIST_LATEST,
                    media_content_id=media_content_id,
                    title="Latest Releases",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )

        elif media_content_type == MEDIA_TYPE_ARTIST_ALBUMS:
            # Artist Albums
            if ":" in media_content_id:
                artist_id = int(media_content_id.split(":")[1])
                artist, albums, featured_albums = await self.hass.async_add_executor_job(self._api.get_artist_albums, artist_id)
                
                children = []
                # Add only regular albums (not featured albums)
                for album in albums:
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(album.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(album),
                            can_play=False,
                            can_expand=True,
                            thumbnail=album.images.small if album.images else None,
                        )
                    )
                
                return BrowseMedia(
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_ARTIST_ALBUMS,
                    media_content_id=media_content_id,
                    title="Albums",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )

        elif media_content_type == MEDIA_TYPE_ARTIST_SINGLES:
            # Artist Singles
            if ":" in media_content_id:
                artist_id = int(media_content_id.split(":")[1])
                singles_data = await self.hass.async_add_executor_job(self._api.get_artist_singles, artist_id, ARTIST_SINGLES_COUNT)
                
                children = []
                for single_data in singles_data:
                    # Parse raw data to Album object (singles are returned as albums)
                    single = self._api._parse_album(single_data)
                    
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(single.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(single),
                            can_play=False,
                            can_expand=True,
                            thumbnail=single.images.small if single.images else None,
                        )
                    )
                
                return BrowseMedia(
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_ARTIST_SINGLES,
                    media_content_id=media_content_id,
                    title="Singles",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )

        elif media_content_type == MEDIA_TYPE_ARTIST_SONGS:
            # Artist Songs
            if ":" in media_content_id:
                artist_id = int(media_content_id.split(":")[1])
                songs_data, _ = await self.hass.async_add_executor_job(self._api.get_artist_songs, artist_id, ARTIST_SONGS_COUNT)

                children = []
                for song_data in songs_data:
                    # Parse raw data to Track object
                    if isinstance(song_data, dict):
                        song = self._api._parse_track(song_data)
                    else:
                        song = song_data  # Already parsed
                    
                    thumbnail = None
                    if song.album and song.album.images:
                        thumbnail = song.album.images.small or song.album.images.medium or song.album.images.large
                    
                    has_album = getattr(song, "album", None) is not None
                    content_id = f"{song.album.id}:{song.id}" if has_album else str(song.id)
                    content_type = "album_track" if has_album else MediaType.TRACK
                    
                    children.append(
                        BrowseMedia(
                            media_class="track",
                            media_content_id=content_id,
                            media_content_type=content_type,
                            title=self._get_display_name(song),
                            can_play=True,
                            can_expand=False,
                            thumbnail=thumbnail,
                        )
                    )
                
                return BrowseMedia(
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_ARTIST_SONGS,
                    media_content_id=media_content_id,
                    title="Songs",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )

        elif media_content_type == MEDIA_TYPE_ARTIST_RELATED:
            # Artist Related Albums (Featured Albums)
            if ":" in media_content_id:
                artist_id = int(media_content_id.split(":")[1])
                artist, albums, featured_albums = await self.hass.async_add_executor_job(self._api.get_artist_albums, artist_id)
                
                children = []
                # Add only featured albums
                for album in featured_albums:
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(album.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(album),
                            can_play=False,
                            can_expand=True,
                            thumbnail=album.images.small if album.images else None,
                        )
                    )
                
                return BrowseMedia(
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_ARTIST_RELATED,
                    media_content_id=media_content_id,
                    title="Related",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )

        elif media_content_type == MEDIA_TYPE_PODCAST:
            try:
                # Parse limit and skip (default 30)
                limit = 30
                skip = 0
                if media_content_id and ":" in media_content_id:
                    parts = media_content_id.split(":")
                    if len(parts) >= 2:
                        limit = max(1, min(100, int(parts[1] or 30)))
                    if len(parts) >= 3:
                        skip = max(0, int(parts[2] or 0))

                podcasts = await self.hass.async_add_executor_job(self._api.get_podcast_albums, limit, skip)

                # Use selected language for display
                children = []
                for podcast in podcasts:
                    # Filter based on whitelist and blacklist
                    if not self._is_artist_allowed(podcast):
                        continue
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(podcast.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(podcast),
                            can_play=False,
                            can_expand=True,
                            thumbnail=podcast.images.small if podcast.images else None,
                        )
                    )

                return BrowseMedia(
                    title="Latest Podcasts",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_PODCAST,
                    media_content_id=MEDIA_TYPE_PODCAST,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="album",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading podcasts: {e}")
                raise HomeAssistantError(f"Error loading podcasts: {e}")

        elif media_content_type == MEDIA_TYPE_KIDS:
            try:
                # Parse limit and skip (default 30)
                limit = 30
                skip = 0
                if media_content_id and ":" in media_content_id:
                    parts = media_content_id.split(":")
                    if len(parts) >= 2:
                        limit = max(1, min(100, int(parts[1] or 30)))
                    if len(parts) >= 3:
                        skip = max(0, int(parts[2] or 0))

                kids_albums = await self.hass.async_add_executor_job(self._api.get_kids_albums, limit, skip)

                # Use selected language for display
                children = []
                for album in kids_albums:
                    # Filter based on whitelist and blacklist
                    if not self._is_artist_allowed(album):
                        continue
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(album.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(album),
                            can_play=False,
                            can_expand=True,
                            thumbnail=album.images.small if album.images else None,
                        )
                    )

                return BrowseMedia(
                    title="Latest Kids",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_KIDS,
                    media_content_id=MEDIA_TYPE_KIDS,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="album",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading kids content: {e}")
                raise HomeAssistantError(f"Error loading kids content: {e}")

        elif media_content_type == MEDIA_TYPE_PODCAST_HOSTS:
            try:
                # Get podcast hosts 
                podcast_hosts = await self.hass.async_add_executor_job(self._api.get_podcasts_hosts, 100, 0)
                
                # Use selected language for display
                children = []
                
                for host in podcast_hosts:
                    # Filter based on whitelist and blacklist
                    if not self._is_artist_allowed(host):
                        continue
                    children.append(
                        BrowseMedia(
                            media_class="artist",
                            media_content_id=str(host.id),
                            media_content_type=MEDIA_TYPE_PODCAST_HOST,
                            title=self._get_display_name(host),
                            can_play=False,
                            can_expand=True,
                            thumbnail=host.images.small if host.images else None,
                        )
                    )

                return BrowseMedia(
                    title="Podcast Hosts",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_PODCAST_HOSTS,
                    media_content_id=MEDIA_TYPE_PODCAST_HOSTS,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="artist",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading podcast hosts: {e}")
                raise HomeAssistantError(f"Error loading podcast hosts: {e}")

        elif media_content_type == MEDIA_TYPE_PODCAST_HOST:
            if media_content_id and media_content_id.isdigit():
                try:
                    host_id = int(media_content_id)
                    
                    # Use the new get_artist_podcasts function
                    artist_data, featured_podcasts, podcasts, total_count = await self.hass.async_add_executor_job(
                        self._api.get_artist_podcasts, host_id, 50, 0, True
                    )
                    
                    if not artist_data:
                        raise HomeAssistantError(f"Podcast host {host_id} not found")

                    # Get host display name
                    host_name = artist_data.get('enName') or artist_data.get('heName') or f"Host {host_id}"

                    children = []
                    
                    # Add featured podcasts first if any
                    for album in featured_podcasts:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(album):
                            continue
                            
                        children.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(album.id),
                                media_content_type=MEDIA_TYPE_ALBUM,
                                title=f"⭐ {self._get_display_name(album)}",  # Star for featured
                                can_play=True,
                                can_expand=True,
                                thumbnail=album.images.small if album.images else None,
                            )
                        )
                    
                    # Add regular podcasts
                    for album in podcasts:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(album):
                            continue
                            
                        children.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(album.id),
                                media_content_type=MEDIA_TYPE_ALBUM,
                                title=self._get_display_name(album),
                                can_play=True,
                                can_expand=True,
                                thumbnail=album.images.small if album.images else None,
                            )
                        )

                    return BrowseMedia(
                        title=f"{host_name} - Podcasts ({total_count})",
                        media_class="directory",
                        media_content_type=MEDIA_TYPE_PODCAST_HOST,
                        media_content_id=str(host_id),
                        can_play=False,
                        can_expand=True,
                        children=children,
                        children_media_class="album",
                    )

                except Exception as e:
                    _LOGGER.error(f"Error loading podcast host content: {e}")
                    raise HomeAssistantError(f"Error loading podcast host content: {e}")

        elif media_content_type == MEDIA_TYPE_KIDS_ARTISTS:
            try:
                # Get kids artists 
                kids_artists = await self.hass.async_add_executor_job(self._api.get_kids_artists, 100, 0)
                
                # Use selected language for display
                children = []
                
                for artist in kids_artists:
                    # Filter based on whitelist and blacklist
                    if not self._is_artist_allowed(artist):
                        continue
                    children.append(
                        BrowseMedia(
                            media_class="artist",
                            media_content_id=str(artist.id),
                            media_content_type=MEDIA_TYPE_KIDS_ARTIST,
                            title=self._get_display_name(artist),
                            can_play=False,
                            can_expand=True,
                            thumbnail=artist.images.small if artist.images else None,
                        )
                    )

                return BrowseMedia(
                    title="Kids Artists",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_KIDS_ARTISTS,
                    media_content_id=MEDIA_TYPE_KIDS_ARTISTS,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="artist",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading kids artists: {e}")
                raise HomeAssistantError(f"Error loading kids artists: {e}")

        elif media_content_type == MEDIA_TYPE_KIDS_ARTIST:
            if media_content_id and media_content_id.isdigit():
                try:
                    artist_id = int(media_content_id)
                    # Get artist info for title
                    kids_artists = await self.hass.async_add_executor_job(self._api.get_kids_artists, 100, 0)
                    artist = next((a for a in kids_artists if a.id == artist_id), None)
                    
                    if not artist:
                        raise HomeAssistantError(f"Kids Artist {artist_id} not found")

                    # Show artist subfolders for kids content
                    children = []
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_KIDS_ARTIST_ALBUMS}:{artist_id}",
                            media_content_type=MEDIA_TYPE_KIDS_ARTIST_ALBUMS,
                            title="Albums",
                            can_play=False,
                            can_expand=True,
                        )
                    )
                    
                    children.append(
                        BrowseMedia(
                            media_class="directory",
                            media_content_id=f"{MEDIA_TYPE_KIDS_ARTIST_LATEST}:{artist_id}",
                            media_content_type=MEDIA_TYPE_KIDS_ARTIST_LATEST,
                            title="Latest",
                            can_play=False,
                            can_expand=True,
                        )
                    )

                    return BrowseMedia(
                        media_class="artist",
                        media_content_id=media_content_id,
                        media_content_type=MEDIA_TYPE_KIDS_ARTIST,
                        title=self._get_display_name(artist),
                        can_play=False,
                        can_expand=True,
                        children=children,
                    )
                except Exception as e:
                    _LOGGER.error(f"Error browsing kids artist {media_content_id}: {e}")
                    raise HomeAssistantError(f"Error browsing kids artist: {e}")

        elif media_content_type == MEDIA_TYPE_KIDS_ARTIST_ALBUMS:
            # Parse artist_id from "kids_artist_albums:123"
            try:
                if ":" in media_content_id:
                    artist_id = int(media_content_id.split(":")[1])
                else:
                    artist_id = int(media_content_id)
                
                albums = await self.hass.async_add_executor_job(self._api.get_kids_artist_albums, artist_id, 50, 0)
                
                # Use selected language for display
                children = []
                for album in albums:
                    children.append(
                        BrowseMedia(
                            media_class="album",
                            media_content_id=str(album.id),
                            media_content_type=MEDIA_TYPE_ALBUM,
                            title=self._get_display_name(album),
                            can_play=False,
                            can_expand=True,
                            thumbnail=album.images.small if album.images else None,
                        )
                    )

                return BrowseMedia(
                    title="Kids Albums",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_KIDS_ARTIST_ALBUMS,
                    media_content_id=media_content_id,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="album",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading kids artist albums: {e}")
                raise HomeAssistantError(f"Error loading kids artist albums: {e}")

        elif media_content_type == MEDIA_TYPE_KIDS_ARTIST_LATEST:
            # Parse artist_id from "kids_artist_latest:123"
            try:
                if ":" in media_content_id:
                    artist_id = int(media_content_id.split(":")[1])
                else:
                    artist_id = int(media_content_id)
                
                tracks = await self.hass.async_add_executor_job(self._api.get_kids_artist_latest, artist_id, 30, 0)
                
                # Use selected language for display
                children = []
                for track in tracks:
                    has_album = getattr(track, "album", None) is not None
                    content_id = f"{track.album.id}:{track.id}" if has_album else str(track.id)
                    content_type = "album_track" if has_album else MediaType.TRACK
                    thumb = (
                        (track.album.images.small or track.album.images.medium or track.album.images.large)
                        if (has_album and getattr(track.album, "images", None)) else None
                    )
                    
                    children.append(
                        BrowseMedia(
                            media_class="track",
                            media_content_id=content_id,
                            media_content_type=content_type,
                            title=self._get_display_name(track),
                            can_play=True,
                            can_expand=False,
                            thumbnail=thumb,
                        )
                    )

                return BrowseMedia(
                    title="Latest Kids Content",
                    media_class="directory",
                    media_content_type=MEDIA_TYPE_KIDS_ARTIST_LATEST,
                    media_content_id=media_content_id,
                    can_play=False,
                    can_expand=True,
                    children=children,
                    children_media_class="track",
                )

            except Exception as e:
                _LOGGER.error(f"Error loading kids artist latest: {e}")
                raise HomeAssistantError(f"Error loading kids artist latest: {e}")

        elif media_content_type == MEDIA_TYPE_FEATURED_PLAYLIST:
            try:
                _LOGGER.debug("[ZING] Fetching featured playlists...")
                playlists = await self.hass.async_add_executor_job(self._api.get_featured_playlists, 0, 100)
                _LOGGER.debug(f"[ZING] Got {len(playlists) if playlists else 0} featured playlists")
                children = []
                
                for playlist in playlists:
                    # Get playlist thumbnail
                    thumbnail = playlist.image if hasattr(playlist, 'image') else None
                    
                    children.append(
                        BrowseMedia(
                            media_class="playlist",
                            media_content_id=str(playlist.id),
                            media_content_type=MEDIA_TYPE_PLAYLIST,
                            title=self._get_display_name(playlist),
                            can_play=True,
                            can_expand=True,
                            thumbnail=thumbnail,
                        )
                    )

                return BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_FEATURED_PLAYLIST,
                    media_content_type=MEDIA_TYPE_FEATURED_PLAYLIST,
                    title="Featured Playlists",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )
            except Exception as e:
                _LOGGER.error("Error browsing featured playlists: %s", e)
                raise HomeAssistantError(f"Error browsing featured playlists: {e}")

        elif media_content_type == MEDIA_TYPE_ARTISTS_PLAYLIST:
            try:
                _LOGGER.debug("[ZING] Fetching artists playlists...")
                playlists = await self.hass.async_add_executor_job(self._api.get_artists_playlist, 0, 100)
                _LOGGER.debug(f"[ZING] Got {len(playlists) if playlists else 0} artists playlists")
                children = []
                
                for playlist in playlists:
                    # Get playlist thumbnail
                    thumbnail = playlist.image if hasattr(playlist, 'image') else None
                    
                    children.append(
                        BrowseMedia(
                            media_class="playlist",
                            media_content_id=str(playlist.id),
                            media_content_type=MEDIA_TYPE_PLAYLIST,
                            title=self._get_display_name(playlist),
                            can_play=True,
                            can_expand=True,
                            thumbnail=thumbnail,
                        )
                    )

                return BrowseMedia(
                    media_class="directory",
                    media_content_id=MEDIA_TYPE_ARTISTS_PLAYLIST,
                    media_content_type=MEDIA_TYPE_ARTISTS_PLAYLIST,
                    title="Artists Playlists",
                    can_play=False,
                    can_expand=True,
                    children=children,
                )
            except Exception as e:
                _LOGGER.error("Error browsing artists playlists: %s", e)
                raise HomeAssistantError(f"Error browsing artists playlists: {e}")

        elif media_content_type == MEDIA_TYPE_PLAYLIST:
            try:
                playlist_id = int(media_content_id)
                tracks = await self.hass.async_add_executor_job(self._api.get_playlist_tracks, playlist_id)
                children = []
                
                for track in tracks:
                    # Get track thumbnail from album
                    thumbnail = None
                    if hasattr(track, 'album') and track.album and hasattr(track.album, 'images') and track.album.images:
                        thumbnail = (
                            track.album.images.small
                            or track.album.images.medium
                            or track.album.images.large
                        )
                    
                    # Encode playlist context in the track ID: "playlist:playlist_id:track_id"
                    playlist_track_id = f"playlist:{playlist_id}:{track.id}"
                    
                    children.append(
                        BrowseMedia(
                            media_class="track",
                            media_content_id=playlist_track_id,
                            media_content_type=MEDIA_TYPE_TRACK,
                            title=self._get_display_name(track),
                            can_play=True,
                            can_expand=False,
                            thumbnail=thumbnail,
                        )
                    )

                # Get playlist info for the title
                playlist_info = await self.hass.async_add_executor_job(self._api.get_playlist_info, playlist_id)
                playlist_title = self._get_display_name(playlist_info) if playlist_info else f"Playlist {playlist_id}"

                return BrowseMedia(
                    media_class="playlist",
                    media_content_id=str(playlist_id),
                    media_content_type=MEDIA_TYPE_PLAYLIST,
                    title=playlist_title,
                    can_play=True,
                    can_expand=True,
                    children=children,
                )
            except Exception as e:
                _LOGGER.error("Error browsing playlist tracks: %s", e)
                raise HomeAssistantError(f"Error browsing playlist tracks: {e}")

        _LOGGER.warning(f"Unknown media content type or ID: type={media_content_type}, id={media_content_id}")
        return BrowseMedia(
            media_class="directory",
            media_content_id="error",
            media_content_type="error",
            title="Error: Unknown Content",
            can_play=False,
            can_expand=False,
        )

    async def async_select_source(self, source: str) -> None:
        _LOGGER.debug(f"Received select_source (label): {source}")
        if not self._source_map:
            _LOGGER.debug(f"Source map empty, building it now")
            self._source_map = self._build_source_map()
        
        _LOGGER.debug(f"Current source map: {self._source_map}")
        _LOGGER.debug(f"Available sources: {list(self._source_map.keys())}")
        
        # If source not found, rebuild the map once in case entities changed
        if source not in self._source_map:
            _LOGGER.warning(f"Source '{source}' not found in current map, rebuilding source map")
            self._source_map = self._build_source_map()
            _LOGGER.debug(f"Rebuilt source map: {self._source_map}")
        
        # Additional fallback: check if source is an entity_id and find its display name
        if source not in self._source_map:
            # Try to find by entity_id (value) instead of display name (key)
            matching_sources = [key for key, value in self._source_map.items() if value == source]
            if matching_sources:
                actual_source = matching_sources[0]
                _LOGGER.info(f"Found source by entity_id: '{source}' -> '{actual_source}'")
                source = actual_source
            else:
                # Try partial matching for convenience (e.g., "speakers_group" matches "speakers_group (Group of 3)")
                partial_matches = [key for key in self._source_map.keys() if source in key or key.startswith(source)]
                if len(partial_matches) == 1:
                    actual_source = partial_matches[0]
                    _LOGGER.info(f"Found source by partial match: '{source}' -> '{actual_source}'")
                    source = actual_source
                elif len(partial_matches) > 1:
                    _LOGGER.warning(f"Multiple partial matches for '{source}': {partial_matches}")
        
        if source not in self._source_map:
            available_sources = list(self._source_map.keys())
            _LOGGER.error(f"Invalid source attempted: '{source}'. Available sources: {available_sources}")
            raise HomeAssistantError(f"Invalid source selected: {source}. Available sources: {available_sources}")
        
        # Store entity_id internally; store label for UI
        self._attr_source = self._source_map[source]
        self._source_label = source
        
        # Asynchronously update Cast device cache for the new source
        await self._update_cast_device_cache_async(self._attr_source)
        
        _LOGGER.info(f"Target set to: {self._attr_source} (label: {source})")
        self._attach_source_listener()
        self.async_write_ha_state()

    async def async_search_media(self, query: SearchMediaQuery) -> SearchMedia:
        _LOGGER.debug(f"Search media called with query: {query.search_query}, type: {query.media_content_type}")

        results_list = []

        # Check if kids-only mode is enabled
        kids_only = self._is_kids_only_mode()
        
        if kids_only:
            # In kids-only mode, search only kids content
            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_KIDS:
                try:
                    _LOGGER.debug(f"Searching for kids content with query: {query.search_query}")
                    kids_albums = await self.hass.async_add_executor_job(
                        self._api.search_kids_albums, query.search_query, 20
                    )
                    _LOGGER.debug(f"Found {len(kids_albums)} kids content results")
                    for album in kids_albums:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(album):
                            continue
                            
                        # Get artist names
                        artist_names = []
                        if album.artists:
                            artist_names.extend(self._get_display_name(a) for a in album.artists if self._get_display_name(a))
                        if album.featuredArtists:
                            artist_names.extend(self._get_display_name(a) for a in album.featuredArtists if self._get_display_name(a))
                        artist_str = ", ".join(artist_names) if artist_names else "Unknown Artist"
                        
                        _LOGGER.debug(f"Adding kids content result: {self._get_display_name(album)} - {artist_str}")
                        results_list.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(album.id),
                                media_content_type=MEDIA_TYPE_KIDS,
                                title=f"{self._get_display_name(album)} - {artist_str}",
                                can_play=False,
                                can_expand=True,
                                thumbnail=album.images.small if album.images else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching kids content: {e}")
        else:
            # Normal mode - search all content types
            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_ARTIST:
                try:
                    artists = await self.hass.async_add_executor_job(
                        self._api.search_artists, query.search_query, 20
                    )
                    for artist in artists:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(artist):
                            continue
                        results_list.append(
                            BrowseMedia(
                                media_class="artist",
                                media_content_id=str(artist.id),
                                media_content_type=MEDIA_TYPE_ARTIST,
                                title=self._get_display_name(artist),
                                can_play=False,
                                can_expand=True,
                                thumbnail=artist.images.small if artist.images else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching artists: {e}")

            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_TRACK:
                try:
                    tracks = await self.hass.async_add_executor_job(
                        self._api.search_tracks, query.search_query, 20
                    )
                    for track in tracks:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(track):
                            continue
                            
                        # Get artist names from album
                        artist_names = []
                        if track.album and track.album.artists:
                            artist_names.extend(self._get_display_name(a) for a in track.album.artists if self._get_display_name(a))
                        if track.album and track.album.featuredArtists:
                            artist_names.extend(self._get_display_name(a) for a in track.album.featuredArtists if self._get_display_name(a))
                        artist_name = ", ".join(artist_names) if artist_names else "Unknown Artist"
                        
                        # Get album image if available
                        thumbnail = None
                        if track.album and track.album.images:
                            thumbnail = track.album.images.small or track.album.images.medium or track.album.images.large
                        
                        results_list.append(
                            BrowseMedia(
                                media_class="track",
                                media_content_id=str(track.id),
                                media_content_type=MEDIA_TYPE_TRACK,
                                title=f"{self._get_display_name(track)} - {artist_name}",
                                can_play=True,
                                can_expand=False,
                                thumbnail=thumbnail,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching tracks: {e}")

            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_ALBUM:
                try:
                    albums = await self.hass.async_add_executor_job(
                        self._api.search_albums, query.search_query, 20
                    )
                    for album in albums:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(album):
                            continue
                            
                        # Get artist names
                        artist_names = []
                        if album.artists:
                            artist_names.extend(self._get_display_name(a) for a in album.artists if self._get_display_name(a))
                        if album.featuredArtists:
                            artist_names.extend(self._get_display_name(a) for a in album.featuredArtists if self._get_display_name(a))
                        
                        artist_name = ", ".join(artist_names) if artist_names else "Unknown Artist"
                        
                        results_list.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(album.id),
                                media_content_type=MEDIA_TYPE_ALBUM,
                                title=f"{self._get_display_name(album)} - {artist_name}",
                                can_play=True,
                                can_expand=True,
                                thumbnail=album.images.small if album.images else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching albums: {e}")

            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_PODCAST:
                try:
                    _LOGGER.debug(f"Searching for podcasts with query: {query.search_query}")
                    podcasts = await self.hass.async_add_executor_job(
                        self._api.search_podcasts, query.search_query, 20
                    )
                    _LOGGER.debug(f"Found {len(podcasts)} podcast results")
                    for podcast in podcasts:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(podcast):
                            continue
                            
                        # Get artist/creator names
                        artist_names = []
                        if podcast.artists:
                            artist_names.extend(self._get_display_name(a) for a in podcast.artists if self._get_display_name(a))
                        if podcast.featuredArtists:
                            artist_names.extend(self._get_display_name(a) for a in podcast.featuredArtists if self._get_display_name(a))
                        artist_str = ", ".join(artist_names) if artist_names else "Unknown Creator"
                        
                        _LOGGER.debug(f"Adding podcast result: {self._get_display_name(podcast)} - {artist_str}")
                        results_list.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(podcast.id),
                                media_content_type=MEDIA_TYPE_PODCAST,
                                title=f"{self._get_display_name(podcast)} - {artist_str}",
                                can_play=False,
                                can_expand=True,
                                thumbnail=podcast.images.small if podcast.images else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching podcasts: {e}")

            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_KIDS:
                try:
                    _LOGGER.debug(f"Searching for kids content with query: {query.search_query}")
                    kids_albums = await self.hass.async_add_executor_job(
                        self._api.search_kids_albums, query.search_query, 20
                    )
                    _LOGGER.debug(f"Found {len(kids_albums)} kids content results")
                    for album in kids_albums:
                        # Filter based on whitelist and blacklist
                        if not self._is_artist_allowed(album):
                            continue
                            
                        # Get artist names
                        artist_names = []
                        if album.artists:
                            artist_names.extend(self._get_display_name(a) for a in album.artists if self._get_display_name(a))
                        if album.featuredArtists:
                            artist_names.extend(self._get_display_name(a) for a in album.featuredArtists if self._get_display_name(a))
                        artist_str = ", ".join(artist_names) if artist_names else "Unknown Artist"
                        
                        _LOGGER.debug(f"Adding kids content result: {self._get_display_name(album)} - {artist_str}")
                        results_list.append(
                            BrowseMedia(
                                media_class="album",
                                media_content_id=str(album.id),
                                media_content_type=MEDIA_TYPE_KIDS,
                                title=f"{self._get_display_name(album)} - {artist_str}",
                                can_play=False,
                                can_expand=True,
                                thumbnail=album.images.small if album.images else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching kids content: {e}")

            # Search playlists if no specific type requested or if playlist type requested
            if query.media_content_type is None or query.media_content_type == MEDIA_TYPE_PLAYLIST:
                try:
                    _LOGGER.debug(f"Searching for playlists with query: {query.search_query}")
                    playlists = await self.hass.async_add_executor_job(self._api.search_playlists, query.search_query, 20)
                    _LOGGER.debug(f"Found {len(playlists)} playlist results")
                    
                    for playlist in playlists:
                        _LOGGER.debug(f"Adding playlist result: {self._get_display_name(playlist)}")
                        results_list.append(
                            BrowseMedia(
                                media_class="playlist",
                                media_content_id=str(playlist.id),
                                media_content_type=MEDIA_TYPE_PLAYLIST,
                                title=self._get_display_name(playlist),
                                can_play=True,
                                can_expand=True,
                                thumbnail=playlist.image if hasattr(playlist, 'image') else None,
                            )
                        )
                except Exception as e:
                    _LOGGER.error(f"Error searching playlists: {e}")

        self._search_result = BrowseMedia(
            media_class="directory",
            media_content_id=f"search_results_{query.search_query}",
            media_content_type="search_results",
            title=f"Search Results for '{query.search_query}'",
            can_play=False,
            can_expand=True,
            children=results_list,
        )

        return SearchMedia(result=self._search_result)

    async def async_media_pause(self):
        _LOGGER.debug("Pause requested")
        if self._attr_source:
            await self.hass.services.async_call(
                "media_player",
                "media_pause",
                {"entity_id": self._attr_source},
                blocking=True,
            )
        if self._state == STATE_PLAYING and self._media_position_updated_at:
            elapsed = (datetime.now(timezone.utc) - self._media_position_updated_at).total_seconds()
            if elapsed > 0:
                self._media_position = min((self._media_position or 0) + elapsed, self._media_duration or float("inf"))
        self._media_position_updated_at = None
        self._state = STATE_PAUSED
        self.async_write_ha_state()
        self._log_now_playing("media_pause")

    async def async_media_play(self):
        _LOGGER.debug("Play requested")
        if self._attr_source:
            await self.hass.services.async_call(
                "media_player",
                "media_play",
                {"entity_id": self._attr_source},
                blocking=True,
            )
        if not self._media_position_updated_at:
            self._media_position_updated_at = datetime.now(timezone.utc)
        self._state = STATE_PLAYING
        self.async_write_ha_state()
        self._log_now_playing("media_play")

    async def async_media_stop(self):
        _LOGGER.debug("Stop requested")
        if self._attr_source:
            await self.hass.services.async_call(
                "media_player",
                "media_stop",
                {"entity_id": self._attr_source},
                blocking=True,
            )
        self._state = STATE_IDLE
        self._media_position_updated_at = None
        self.async_write_ha_state()

    async def async_turn_on(self):
        _LOGGER.debug("Turn on requested")
        # Turn on the selected source device if available
        if self._attr_source:
            await self.hass.services.async_call(
                "media_player",
                "turn_on",
                {"entity_id": self._attr_source},
                blocking=True,
            )
        # Set state to idle (ready to play)
        self._state = STATE_IDLE
        self.async_write_ha_state()

    async def async_turn_off(self):
        _LOGGER.debug("Turn off requested")
        # Turn off the selected source device if available
        if self._attr_source:
            await self.hass.services.async_call(
                "media_player",
                "turn_off",
                {"entity_id": self._attr_source},
                blocking=True,
            )
        # Set state to off
        self._state = STATE_OFF
        # Clear current media info
        self._current_track = None
        self._media_url = None
        self._media_image_url = None
        self._media_artist = None
        self._media_title = None
        self._media_album_name = None
        self._media_position = 0.0
        self._media_position_updated_at = None
        self._media_duration = None
        # Detach source listener
        self._detach_source_listener()
        self.async_write_ha_state()

    async def async_media_next_track(self):
        # Prevent rapid successive calls
        if hasattr(self, '_next_track_in_progress') and self._next_track_in_progress:
            _LOGGER.debug("[ZING] Next track already in progress, skipping")
            return
        
        self._next_track_in_progress = True
        try:
            # Repeat ONE: restart current track
            if self._repeat == "one":
                if self._current_track:
                    await self._play_track(self._current_track)
                return

            track = await self.hass.async_add_executor_job(self._playback_controller.play_next)
            if track:
                _LOGGER.debug(f"[ZING] Playing next: {track.enName} (ID: {track.id})")
                await self._play_track(track)
            else:
                if self._repeat == "all":
                    # loop album
                    if self._playback_controller.playlist:
                        self._playback_controller.current_index = 0
                        track = self._playback_controller.get_current_track()
                        if track:
                            await self._play_track(track)
                            return
                _LOGGER.debug("[ZING] No next track")
        finally:
            self._next_track_in_progress = False

    async def async_media_previous_track(self):
        # If we're not far in, previous should move to prior track;
        # otherwise restart current. We'll keep it simple and go previous.
        track = await self.hass.async_add_executor_job(self._playback_controller.play_previous)
        if track:
            await self._play_track(track)

    # ---------- Autonext helpers ----------

    def _get_cached_cast_device_status(self, entity_id: str) -> bool:
        """Get cached Cast device status, detecting if not cached."""
        if entity_id not in self._is_cast_device_cache:
            # Perform detection and cache result
            self._is_cast_device_cache[entity_id] = self._detect_cast_device_sync(entity_id)
        return self._is_cast_device_cache[entity_id]

    def _detect_cast_device_sync(self, entity_id: str) -> bool:
        """Fast synchronous Cast device detection using minimal checks."""
        if not entity_id:
            return False
        
        # Method 1: Quick entity registry platform check
        try:
            entity_registry = er.async_get(self.hass)
            entity_entry = entity_registry.async_get(entity_id)
            if entity_entry and entity_entry.platform == "cast":
                _LOGGER.debug(f"[ZING] Fast Cast detection via platform: {entity_id}")
                return True
        except Exception:
            pass
        
        # Method 2: Check entity state attributes for Cast-specific markers
        source_state = self.hass.states.get(entity_id)
        if source_state:
            attributes = source_state.attributes
            
            # Cast devices have app_id or app_name
            if attributes.get("app_id") is not None or attributes.get("app_name") is not None:
                _LOGGER.debug(f"[ZING] Fast Cast detection via app attributes: {entity_id}")
                return True
            
            # Check for Cast-specific metadata structure
            media_metadata = attributes.get("media_metadata")
            if isinstance(media_metadata, dict) and any(key in media_metadata for key in ["title", "artist", "albumName"]):
                friendly_name = attributes.get("friendly_name", "").lower()
                if any(indicator in friendly_name for indicator in ["chromecast", "google home", "nest"]):
                    _LOGGER.debug(f"[ZING] Fast Cast detection via metadata + name: {entity_id}")
                    return True
        
        return False

    async def _update_cast_device_cache_async(self, entity_id: str):
        """Asynchronously update Cast device cache with full detection."""
        if not entity_id:
            return
        
        try:
            device_registry = dr.async_get(self.hass)
            entity_registry = er.async_get(self.hass)
            
            entity_entry = entity_registry.async_get(entity_id)
            if entity_entry and entity_entry.device_id:
                device_entry = device_registry.async_get(entity_entry.device_id)
                if device_entry:
                    # Check device identifiers for Cast domain
                    for identifier_domain, _ in device_entry.identifiers:
                        if identifier_domain == "cast":
                            self._is_cast_device_cache[entity_id] = True
                            _LOGGER.debug(f"[ZING] Async Cast detection via device registry: {entity_id}")
                            return
                    
                    # Check manufacturer for Google Cast devices
                    if device_entry.manufacturer:
                        cast_manufacturers = ["Google", "Google Inc.", "Google LLC"]
                        if any(mfg in device_entry.manufacturer for mfg in cast_manufacturers):
                            if device_entry.model and any(
                                model_term in device_entry.model.lower() 
                                for model_term in ["chromecast", "google home", "nest", "cast"]
                            ):
                                self._is_cast_device_cache[entity_id] = True
                                _LOGGER.debug(f"[ZING] Async Cast detection via manufacturer/model: {entity_id}")
                                return
            
            # If we get here, it's not a Cast device
            if entity_id not in self._is_cast_device_cache:
                self._is_cast_device_cache[entity_id] = False
        
        except Exception as e:
            _LOGGER.debug(f"[ZING] Async Cast detection failed for {entity_id}: {e}")
            # Fallback to cached sync detection result
            pass

    def _attach_source_listener(self):
        # remove previous if any
        if self._unsub_source_listener:
            self._unsub_source_listener()
            self._unsub_source_listener = None

        if not self._attr_source:
            return

        # Determine which entities to monitor
        entities_to_monitor = [self._attr_source]
        
        # If source is a group, also monitor group members for better state tracking
        if self._is_speaker_group(self._attr_source):
            group_state = self.hass.states.get(self._attr_source)
            if group_state:
                group_members = group_state.attributes.get("entity_id", [])
                if group_members:
                    entities_to_monitor.extend(group_members)
                    _LOGGER.debug(f"[ZING] Monitoring group {self._attr_source} and members: {group_members}")

        async def _on_source_change(event):
            new = event.data.get("new_state")
            if not new:
                return
            
            # Handle state changes from any monitored entity
            entity_id = new.entity_id
            
            # Sync volume state from main source or any group member
            if entity_id == self._attr_source or entity_id in entities_to_monitor[1:]:
                volume_level = new.attributes.get("volume_level")
                if volume_level is not None:
                    try:
                        self._volume_level = float(volume_level)
                    except Exception:
                        pass
                
                is_muted = new.attributes.get("is_volume_muted")
                if is_muted is not None:
                    self._is_volume_muted = bool(is_muted)
                
                # Sync media position from renderer for accuracy (same logic as manual sync)
                renderer_position = new.attributes.get("media_position")
                
                # Always sync position like non-Cast devices - no state restrictions
                if renderer_position is not None:
                    try:
                        new_position = float(renderer_position)
                        current_stored_position = self._media_position  # Just the stored position, no calculation
                        
                        # Debug logging to understand position sync                        
                        # Update very frequently to keep UI in perfect sync (very low threshold)
                        if abs(current_stored_position - new_position) > 0.1:  # 0.1 second threshold for precise sync
                            # Use the renderer's position directly
                            self._media_position = new_position
                            self._media_position_updated_at = datetime.now(timezone.utc)
                            # Force comprehensive state update to sync UI immediately
                            self.async_write_ha_state()
                    except Exception as e:
                        _LOGGER.debug(f"[ZING] Failed to sync Cast position: {e}")
                        pass
                
                # Professional Cast Integration: Immediate Response to State Changes
                # Based on Home Assistant's official Cast integration approach
                # No delays - immediate response to Cast device state transitions
                if new.state in (STATE_IDLE, STATE_OFF) and self._state == STATE_PLAYING:
                    current_position = self.media_position
                    track_duration = self._media_duration
                    
                    # Professional approach: Smart detection without arbitrary delays
                    should_advance = False
                    
                    if current_position and track_duration:
                        # If we have position data, use it for precise detection
                        progress_percentage = (current_position / track_duration) * 100
                        if progress_percentage >= 90:
                            should_advance = True
                            _LOGGER.debug(f"[ZING] Track {progress_percentage:.1f}% complete - advancing immediately")
                        else:
                            _LOGGER.debug(f"[ZING] Track only {progress_percentage:.1f}% complete - not advancing")
                    else:
                        # Use cached Cast device detection (non-blocking)
                        is_cast_device = self._get_cached_cast_device_status(self._attr_source)
                        
                        if is_cast_device:
                            # Cast devices: Follow professional pattern - immediate response
                            # Official Cast integration doesn't use delays for idle detection
                            should_advance = True
                            _LOGGER.debug(f"[ZING] Cast device detected (cached) - advancing immediately")
                        else:
                            # Non-Cast devices without position: Use duration-based heuristic
                            if track_duration:
                                should_advance = True
                                _LOGGER.debug(f"[ZING] Non-Cast device idle with duration - advancing")
                            else:
                                _LOGGER.debug(f"[ZING] No position/duration data - not advancing")
                    
                    if should_advance:
                        # Professional track start time safeguard: Prevent auto-advance for first 3 seconds
                        # This prevents advancing during track transitions and loading delays
                        current_time = time.time()
                        if hasattr(self, '_track_start_time') and self._track_start_time:
                            time_since_start = current_time - self._track_start_time
                            if time_since_start < 3.0:  # 3-second grace period
                                _LOGGER.debug(f"[ZING] Track too new ({time_since_start:.1f}s) - not advancing")
                                should_advance = False
                        
                        if should_advance:
                            # Professional approach: Immediate advancement without delays
                            # This matches how the official Cast integration handles track endings
                            await self.async_media_next_track()
                
                # Update state to reflect any changes
                self.async_write_ha_state()

        self._unsub_source_listener = async_track_state_change_event(
            self.hass, entities_to_monitor, _on_source_change
        )
