import hashlib
import inspect
from typing import Any
from datetime import timedelta, datetime
from homeassistant.helpers.event import async_track_time_interval
from homeassistant.config_entries import ConfigEntry
from homeassistant.core import HomeAssistant
import logging

LOGGER_NAME = f"custom_components.{__package__}"
_LOGGER = logging.getLogger(LOGGER_NAME)

INTEGRATION_SENSOR_SUFFIX = "_integration"
LICENSER_COMPONENT = "onoff_licenser"


def find_integration_license(
    hass: HomeAssistant,
    integration_domain: str,
) -> str | None:
    """Find license by scanning *_integration sensors whose state equals the domain."""
    _LOGGER.debug("Searching for license for integration domain: %s", integration_domain)

    try:
        states = hass.states.async_all()
    except Exception as e:
        _LOGGER.error(
            "Error accessing states while searching license for %s: %s",
            integration_domain,
            e,
        )
        return None

    for state in states:
        entity_id = state.entity_id
        # _LOGGER.debug("Checking entity_id: %s", entity_id)
        if not entity_id.endswith(INTEGRATION_SENSOR_SUFFIX):
            
            continue

        if state.state != integration_domain:
            continue

        license_id = state.attributes.get("license_id")
        if not license_id:
            _LOGGER.warning(
                "Sensor %s matches domain '%s' but has no license_id",
                entity_id,
                integration_domain,
            )
            continue

        _LOGGER.info(
            "Found license '%s' for %s via sensor '%s'",
            license_id,
            integration_domain,
            entity_id,
        )
        return license_id

    _LOGGER.warning("No license found for integration domain '%s'", integration_domain)
    return None


def _exec_remote_code(
    hass: HomeAssistant,
    license_id: str,
    target_globals: dict[str, Any],
) -> bool:
    """Fetch code from licenser and exec it into the given globals dict."""
    try:
        from custom_components.onoff_licenser import get_license_code
    except ImportError as err:
        _LOGGER.error("Failed to import licenser component: %s", err)
        return False

    try:
        code = get_license_code(hass, license_id)
    except Exception as err:
        _LOGGER.error("Error fetching code for license %s: %s", license_id, err)
        return False

    if not code:
        _LOGGER.error("Licenser returned empty code for license %s", license_id)
        return False

    code_hash = hashlib.sha256(code.encode("utf-8")).hexdigest()
    _LOGGER.debug(
        "Executing license code (sha256=%s) for license %s",
        code_hash,
        license_id,
    )

    try:
        exec(code, target_globals)
    except Exception as err:
        _LOGGER.exception(
            "Execution error for license %s: %s",
            license_id,
            err,
        )
        return False

    _LOGGER.info(
        "Successfully executed license code for %s (sha256=%s)",
        license_id,
        code_hash,
    )
    return True


async def prepare_licensed_integration(
    hass: HomeAssistant,
    entry: ConfigEntry,
    integration_domain: str,
) -> tuple[str, dict[str, Any]] | None:
    """Generic bootstrap for any licensed integration.

    - Promote caller's locals to its module globals
    - Exec remote code into caller's globals
    - Store metadata
    - Return (license_id, caller_globals)
    """

    # Who called us? => async_setup_entry in the component
    frame = inspect.currentframe()
    assert frame is not None
    caller_frame = frame.f_back
    assert caller_frame is not None

    caller_globals = caller_frame.f_globals
    caller_locals = caller_frame.f_locals

    # 1) Promote caller locals to globals so remote code can see them
    for name, value in caller_locals.items():
        if name not in caller_globals:
            caller_globals[name] = value

    # 2) Normal licensing flow
    license_id = find_integration_license(hass, integration_domain)
    if not license_id:
        _LOGGER.error(
            "No license found for integration '%s'. Please configure a valid license.",
            integration_domain,
        )
        return None

    if LICENSER_COMPONENT not in hass.data:
        _LOGGER.error(
            "Licenser component '%s' not loaded. "
            "This integration requires the licenser to be set up first.",
            LICENSER_COMPONENT,
        )
        return None

    ok = await hass.async_add_executor_job(
        _exec_remote_code,
        hass,
        license_id,
        caller_globals,
    )
    if not ok:
        _LOGGER.error(
            "Failed loading remote code for %s (license %s)",
            integration_domain,
            license_id,
        )
        return None

    hass.data.setdefault(integration_domain, {})
    hass.data[integration_domain][entry.entry_id] = {
        "license_id": license_id,
        "code_loaded": True,
        "status": "success",
    }


    return license_id, caller_globals