# custom_components/zing_music/config_flow.py

from __future__ import annotations

import voluptuous as vol
from typing import Any, Optional
import logging

from homeassistant import config_entries
from homeassistant.core import callback, HomeAssistant
from homeassistant.helpers.selector import selector
from homeassistant.helpers import config_validation as cv
from homeassistant.const import CONF_EMAIL, CONF_PASSWORD

_LOGGER = logging.getLogger(__name__)

from .const import DOMAIN, CONF_ALLOWED_ENTITY_IDS, CONF_LATEST_COUNT, CONF_KIDS_ONLY, CONF_RESTRICTED_ARTISTS, CONF_WHITELISTED_ARTISTS, DEFAULT_LATEST_COUNT, LANG_ENGLISH, LANG_HEBREW, CONF_LANGUAGE
from .api import ZingMusicUser, ZingMusicUserError

class ZingMusicConfigFlow(config_entries.ConfigFlow, domain=DOMAIN):
    VERSION = 1

    async def async_step_user(self, user_input: Optional[dict[str, Any]] = None):
        errors = {}
        
        if user_input is not None:
            email = user_input.get(CONF_EMAIL)
            password = user_input.get(CONF_PASSWORD)
            
            try:
                # Attempt authentication
                user_manager = await self.hass.async_add_executor_job(
                    ZingMusicUser, email, password, True
                )
                
                # Check subscription status
                if not user_manager.subscribed:
                    _LOGGER.warning(f"User {email} authenticated but not subscribed.")
                    # We proceed but maybe note it? For now just log.
                
                title = (user_input.get("name") or f"Zing Music Player ({email})").strip()
                
                return self.async_create_entry(title=title, data={
                    CONF_EMAIL: email,
                    CONF_PASSWORD: password
                })
                
            except ZingMusicUserError as e:
                _LOGGER.error(f"Authentication error: {e}")
                errors["base"] = "invalid_auth"
            except Exception as e:
                _LOGGER.exception(f"Unexpected error: {e}")
                errors["base"] = "unknown"

        default_name = f"Zing Music Player #{len(self._async_current_entries()) + 1}"
        
        schema = vol.Schema({
            vol.Required(CONF_EMAIL): str,
            vol.Required(CONF_PASSWORD): str,
            vol.Optional("name", default=default_name): str,
        })
        
        return self.async_show_form(step_id="user", data_schema=schema, errors=errors)

    @staticmethod
    @callback
    def async_get_options_flow(config_entry):
        return ZingMusicOptionsFlowHandler(config_entry)


class ZingMusicOptionsFlowHandler(config_entries.OptionsFlow):
    def __init__(self, config_entry: config_entries.ConfigEntry):
        self._entry = config_entry

    async def _get_available_media_players(self) -> list[str]:
        """Get all media_player entities excluding Music Assistant ones."""
        from homeassistant.helpers import entity_registry as er
        
        # Get entity registry
        entity_registry = er.async_get(self.hass)
        
        # Get Music Assistant entities to exclude
        music_assistant_entities = set()
        try:
            for entity in entity_registry.entities.values():
                if entity.platform == "music_assistant":
                    music_assistant_entities.add(entity.entity_id)
            _LOGGER.debug(f"[ZING CONFIG] Found {len(music_assistant_entities)} Music Assistant entities to exclude: {music_assistant_entities}")
        except Exception as e:
            _LOGGER.debug(f"Could not get Music Assistant entities: {e}")
        
        # Get all media_player states
        all_media_players = []
        for state in self.hass.states.async_all("media_player"):
            # Skip Music Assistant entities (by platform)
            if state.entity_id in music_assistant_entities:
                continue
            # Skip Music Assistant entities (by app_id)
            if state.attributes.get('app_id') == 'music_assistant':
                _LOGGER.debug(f"[ZING CONFIG] Excluding entity {state.entity_id} due to app_id: music_assistant")
                continue
            supported_features = state.attributes.get('supported_features', 0)
            if not (supported_features & 512):  # MediaPlayerEntityFeature.PLAY_MEDIA = 512
                continue
            all_media_players.append(state.entity_id)
        
        return all_media_players

    async def async_step_init(self, user_input: Optional[dict[str, Any]] = None):
        errors = {}
        
        if user_input is not None:
            _LOGGER.debug(f"[ZING CONFIG] Received user input: {user_input}")

            # If email/password are provided (and changed), verify them
            new_email = user_input.get(CONF_EMAIL)
            new_password = user_input.get(CONF_PASSWORD)
            
            # If creds changed, verify them
            if new_email and new_password:
                 try:
                    await self.hass.async_add_executor_job(
                        ZingMusicUser, new_email, new_password, True
                    )
                 except ZingMusicUserError:
                     errors["base"] = "invalid_auth"
                     # Redisplay form with error
                     return await self._show_options_form(user_input, errors)

            # Update entry data for credentials (sensitive info usually in data, not options, but here mixed usage)
            # We must update the main config entry data for credentials if they changed
            if new_email and new_password:
                self.hass.config_entries.async_update_entry(
                    self._entry, 
                    data={**self._entry.data, CONF_EMAIL: new_email, CONF_PASSWORD: new_password}
                )

            new_options = self._entry.options.copy()
            
            # Process other options
            if CONF_RESTRICTED_ARTISTS not in user_input:
                new_options[CONF_RESTRICTED_ARTISTS] = []
            else:
                restricted_input = user_input.get(CONF_RESTRICTED_ARTISTS, "")
                if restricted_input and restricted_input.strip():
                    restricted_list = [name.strip() for name in str(restricted_input).split(",") if name.strip()]
                else:
                    restricted_list = []
                new_options[CONF_RESTRICTED_ARTISTS] = restricted_list

            if CONF_WHITELISTED_ARTISTS not in user_input:
                new_options[CONF_WHITELISTED_ARTISTS] = []
            else:
                whitelisted_input = user_input.get(CONF_WHITELISTED_ARTISTS, "")
                if whitelisted_input and whitelisted_input.strip():
                    whitelisted_list = [name.strip() for name in str(whitelisted_input).split(",") if name.strip()]
                else:
                    whitelisted_list = []
                new_options[CONF_WHITELISTED_ARTISTS] = whitelisted_list

            if CONF_LANGUAGE in user_input:
                new_options[CONF_LANGUAGE] = user_input[CONF_LANGUAGE]
            if CONF_ALLOWED_ENTITY_IDS in user_input:
                new_options[CONF_ALLOWED_ENTITY_IDS] = user_input[CONF_ALLOWED_ENTITY_IDS]
            if CONF_KIDS_ONLY in user_input:
                new_options[CONF_KIDS_ONLY] = user_input[CONF_KIDS_ONLY]
            if CONF_LATEST_COUNT in user_input:
                new_options[CONF_LATEST_COUNT] = user_input[CONF_LATEST_COUNT]

            _LOGGER.debug(f"[ZING CONFIG] Saving options: {new_options}")
            return self.async_create_entry(title="", data=new_options)

        return await self._show_options_form(None, errors)

    async def _show_options_form(self, user_input, errors):
        # Get current restricted artists and format as a string for the input box.
        current_restricted = self._entry.options.get(CONF_RESTRICTED_ARTISTS, [])
        if isinstance(current_restricted, list):
            restricted_default = ",".join(current_restricted)
        else:
            restricted_default = ""

        current_whitelisted = self._entry.options.get(CONF_WHITELISTED_ARTISTS, [])
        if isinstance(current_whitelisted, list):
            whitelisted_default = ",".join(current_whitelisted)
        else:
            whitelisted_default = ""
        
        # Get current credentials from data
        current_email = self._entry.data.get(CONF_EMAIL, "")
        
        # Get all media_player entities excluding Music Assistant ones
        available_entities = await self._get_available_media_players()
        
        schema = vol.Schema({
            # Credentials (Optional update)
            vol.Required(CONF_EMAIL, default=current_email): str,
            vol.Required(CONF_PASSWORD, default=self._entry.data.get(CONF_PASSWORD, "")): str,

            # Language settings
            vol.Optional(
                CONF_LANGUAGE,
                default=self._entry.options.get(CONF_LANGUAGE, LANG_ENGLISH),
            ): vol.In([LANG_ENGLISH, LANG_HEBREW]),
            
            # Allowed Entities
            vol.Optional(
                CONF_ALLOWED_ENTITY_IDS,
                default=self._entry.options.get(CONF_ALLOWED_ENTITY_IDS, []),
            ): selector({
                "entity": {
                    "domain": "media_player",
                    "multiple": True,
                    "include_entities": available_entities,
                }
            }),
            
            # Latest Items Count
            vol.Required(
                CONF_LATEST_COUNT,
                default=self._entry.options.get(CONF_LATEST_COUNT, DEFAULT_LATEST_COUNT),
            ): selector({
                "number": {
                    "min": 1,
                    "max": 100,
                    "mode": "box"
                }
            }),
            
            # Restrictions - Kids checkbox
            vol.Optional(
                CONF_KIDS_ONLY,
                default=self._entry.options.get(CONF_KIDS_ONLY, False),
            ): bool,
            
            # Restrictions - Restricted Artists
            vol.Optional(
                CONF_RESTRICTED_ARTISTS,
                description={"suggested_value": restricted_default} 
            ): selector({"text": {}}),
            
            # Restrictions - Whitelisted Artists
            vol.Optional(
                CONF_WHITELISTED_ARTISTS,
                description={"suggested_value": whitelisted_default} 
            ): selector({"text": {}}),
        })

        return self.async_show_form(step_id="init", data_schema=schema, errors=errors)