# custom_components/zing_music/api.py

import requests
import json
import logging
from dataclasses import dataclass
from typing import List, Literal, Optional, Tuple, Dict, Any
import random

_LOGGER = logging.getLogger(__name__)

# =========================
# 📦 Data Models
# =========================

@dataclass
class ImageSet:
    small: Optional[str] = None
    medium: Optional[str] = None
    large: Optional[str] = None


@dataclass
class Artist:
    id: int
    enName: str
    heName: str
    images: Optional[ImageSet] = None


@dataclass
class Album:
    id: int
    enName: str
    heName: str
    releasedAt: Optional[str]
    images: Optional[ImageSet]
    premium: Optional[bool] = None
    albumType: Optional[str] = None
    rssUrl: Optional[str] = None
    artists: Optional[List[Artist]] = None
    featuredArtists: Optional[List[Artist]] = None
    tracks: Optional[List["Track"]] = None


@dataclass
class TrackCredit:
    id: int
    enName: str
    heName: str
    role: str


@dataclass
class Track:
    id: int
    enName: str
    heName: str
    fileName: str
    duration: Optional[int]
    enLyrics: Optional[str]
    heLyrics: Optional[str]
    enDesc: Optional[str]
    heDesc: Optional[str]
    credits: List[TrackCredit]
    album: Album
    trackNumber: Optional[int] = None
    is_last_track: Optional[bool] = None 
    download_url: Optional[str] = None


@dataclass
class Playlist:
    id: int
    name: str
    enName: str
    heName: str
    image: Optional[str] = None
    private: Optional[bool] = None
    featured: Optional[bool] = None
    premium: Optional[bool] = None
    user: Optional[dict] = None

@dataclass
class UserInfo:
    uid: str
    email: str
    display_name: Optional[str] = None
    subscribed: bool = False

class ZingMusicUserError(Exception):
    """Custom exception for ZingMusicUser errors"""
    pass

class ZingMusicUser:
    """Firebase authentication and user management for Zing Music"""
    
    # Class constants
    FIREBASE_API_KEY = "AIzaSyByyRIbRvi6MLOjfWqdv73B88x2QsVkOZA"
    FIREBASE_PROJECT_ID = "1:1033576174882:web:c029594023622c57cb017b"
    LOGIN_URL = "https://identitytoolkit.googleapis.com/v1/accounts:signInWithPassword"
    LOOKUP_URL = "https://identitytoolkit.googleapis.com/v1/accounts:lookup"
    
    def __init__(self, email: str, password: str, auto_login: bool = True, graphql_url: str = None):
        self.email = email
        self.password = password
        self.graphql_url = graphql_url
        
        # Initialize user properties
        self.bearer_token: Optional[str] = None
        self.user_info: Optional[UserInfo] = None
        self._authenticated = False
        
        # Temporary storage
        self._temp_uid = None
        self._temp_display_name = None
        
        if auto_login:
            self.authenticate()

    @property
    def headers(self) -> Dict[str, str]:
        """Firebase API headers"""
        return {
            "accept": "*/*",
            "content-type": "application/json",
            "x-client-version": "Chrome/JsCore/11.9.1/FirebaseCore-web",
            "x-firebase-gmpid": self.FIREBASE_PROJECT_ID
        }

    @property
    def auth_headers(self) -> Dict[str, str]:
        """Headers with bearer token for authenticated requests"""
        if not self.bearer_token:
            return {}
        return {"Authorization": f"Bearer {self.bearer_token}"}

    @property
    def is_authenticated(self) -> bool:
        """Check if user is authenticated"""
        return self._authenticated and self.bearer_token is not None

    @property
    def uid(self) -> Optional[str]:
        """Get user ID"""
        return self.user_info.uid if self.user_info else None

    @property
    def subscribed(self) -> bool:
        """Check if user has active subscription"""
        return self.user_info.subscribed if self.user_info else False

    def authenticate(self) -> bool:
        """Complete authentication flow"""
        try:
            # Step 1: Login and get bearer token
            if not self._login():
                return False
                
            # Step 2: Get user details
            if not self._fetch_user_details():
                return False
                
            # Step 3: Check subscription status
            subscription_status = self._get_subscription_status()
            
            # Create user info object
            self.user_info = UserInfo(
                uid=self._temp_uid,
                email=self.email,
                display_name=self._temp_display_name,
                subscribed=subscription_status
            )
            
            self._authenticated = True
            _LOGGER.info(f"Successfully authenticated user: {self.email}, Subscribed: {subscription_status}")
            return True
            
        except Exception as e:
            _LOGGER.error(f"Authentication failed: {e}")
            self._reset_auth_state()
            raise # Re-raise for config flow handling

    def _login(self) -> bool:
        """Step 1: Login with email/password"""
        payload = {
            "returnSecureToken": True,
            "email": self.email,
            "password": self.password,
            "clientType": "CLIENT_TYPE_WEB"
        }

        try:
            response = requests.post(
                f"{self.LOGIN_URL}?key={self.FIREBASE_API_KEY}", 
                headers=self.headers, 
                json=payload,
                timeout=10
            )
            
            if response.status_code == 200:
                self.bearer_token = response.json().get("idToken")
                return self.bearer_token is not None
            elif response.status_code == 400:
                error_data = response.json()
                error_message = error_data.get("error", {}).get("message", "Invalid credentials")
                raise ZingMusicUserError(f"Login failed: {error_message}")
            else:
                raise ZingMusicUserError(f"Login failed: HTTP {response.status_code}")
                
        except requests.RequestException as e:
            raise ZingMusicUserError(f"Network error during login: {e}")

    def _fetch_user_details(self) -> bool:
        """Step 2: Get user details from Firebase"""
        if not self.bearer_token:
            return False

        payload = {"idToken": self.bearer_token}

        try:
            response = requests.post(
                f"{self.LOOKUP_URL}?key={self.FIREBASE_API_KEY}",
                headers=self.headers,
                json=payload,
                timeout=10
            )
            
            if response.status_code == 200:
                users = response.json().get("users", [])
                if not users:
                    raise ZingMusicUserError("No user data returned")
                    
                user_details = users[0]
                # Store temporarily until we create UserInfo object
                self._temp_uid = user_details.get("localId")
                self._temp_display_name = user_details.get("displayName")
                return True
            else:
                raise ZingMusicUserError(f"User lookup failed: HTTP {response.status_code}")
                
        except requests.RequestException as e:
            raise ZingMusicUserError(f"Network error during user lookup: {e}")

    def _get_subscription_status(self) -> bool:
        """Step 3: Check subscription status via GraphQL"""
        if not self._temp_uid:
            return False
            
        if not self.graphql_url:
            _LOGGER.debug("Skipping subscription check: GraphQL URL not available")
            return False

        query = """
        query GetUser($uid: String!) {
            user(where: {uid: $uid}) {
                subscribed
            }
        }
        """
        
        variables = {"uid": self._temp_uid}
        
        # Use simple headers here because auth header is managed by properties
        headers = {"Authorization": f"Bearer {self.bearer_token}"} if self.bearer_token else {}
        headers.update({"content-type": "application/json"})

        try:
            response = requests.post(
                self.graphql_url,
                json={"query": query, "variables": variables},
                headers=headers,
                timeout=10
            )
            response.raise_for_status()
            
            data = response.json()
            return data.get("data", {}).get("user", {}).get("subscribed", False)
            
        except requests.RequestException as e:
            _LOGGER.warning(f"Could not fetch subscription status: {e}")
            return False  # Default to no subscription on error

    def _reset_auth_state(self):
        """Reset all authentication state"""
        self.bearer_token = None
        self.user_info = None
        self._authenticated = False
        self._temp_uid = None
        self._temp_display_name = None


# =========================
# 🎧 API Class
# =========================

class ZingMusicAPI:
    def __init__(self, email: str = None, password: str = None, endpoint: str = None, headers: dict = None, api_base_url: str = None):
        self.endpoint = endpoint
        self.base_headers = headers or {}
        self.api_base_url = api_base_url
        self._album_cache: dict[int, Album] = {}
        
        self.user_manager = None
        if email and password:
            self.user_manager = ZingMusicUser(email, password, auto_login=False, graphql_url=endpoint)

    def authenticate(self):
        """Perform authentication using the user manager."""
        if self.user_manager:
            self.user_manager.authenticate()

    @property
    def is_authenticated(self) -> bool:
        return self.user_manager and self.user_manager.is_authenticated

    @property
    def is_subscribed(self) -> bool:
        return self.user_manager and self.user_manager.subscribed

    def _post(self, payload: dict) -> dict:
        if not self.endpoint:
            raise ValueError("API endpoint not configured")

        # Merge base headers with auth headers
        request_headers = self.base_headers.copy()
        if self.user_manager and self.user_manager.is_authenticated:
            request_headers.update(self.user_manager.auth_headers)

        response = requests.post(self.endpoint, json=payload, headers=request_headers)
        response.raise_for_status()
        return response.json()["data"]

    # ======================
    # 🧩 Shared Parsers
    # ======================

    def _clean_dict(self, data: dict) -> dict:
        """Remove __typename and other GraphQL metadata from dictionary recursively."""
        if not data:
            return data
        
        cleaned = {}
        for k, v in data.items():
            if k == '__typename':
                continue
            elif isinstance(v, dict):
                cleaned[k] = self._clean_dict(v)
            elif isinstance(v, list):
                cleaned[k] = [self._clean_dict(item) if isinstance(item, dict) else item for item in v]
            else:
                cleaned[k] = v
        
        return cleaned

    def _parse_images(self, imgs: Optional[dict]) -> Optional[ImageSet]:
        if not imgs:
            return None
        clean_imgs = self._clean_dict(imgs)
        return ImageSet(**clean_imgs) if clean_imgs else None

    def _parse_artist(self, data: dict) -> Artist:
        clean_data = self._clean_dict(data)
        
        # Parse images if present
        images = None
        if "images" in clean_data and clean_data["images"]:
            images = ImageSet(
                small=clean_data["images"].get("small"),
                medium=clean_data["images"].get("medium"),
                large=clean_data["images"].get("large")
            )
        
        return Artist(
            id=clean_data.get("id", 0),  # Default to 0 if id is missing
            enName=clean_data["enName"],
            heName=clean_data["heName"],
            images=images
        )

    def _parse_album(self, data: dict) -> Album:
        """
        Parses album data, including nested artist information.
        """
        clean_data = self._clean_dict(data)
        
        # Parse artists and featured artists if they exist in the data
        artists = [self._parse_artist(a) for a in clean_data.get("artists", [])]
        featured_artists = [self._parse_artist(a) for a in clean_data.get("featuredArtists", [])]

        return Album(
            id=clean_data["id"],
            enName=clean_data["enName"],
            heName=clean_data["heName"],
            releasedAt=clean_data.get("releasedAt"),
            images=self._parse_images(clean_data.get("images")),
            premium=clean_data.get("premium"),
            albumType=clean_data.get("albumType"),
            rssUrl=clean_data.get("rssUrl"),
            artists=artists, # Correctly pass the parsed artist list
            featuredArtists=featured_artists, # Correctly pass the parsed featured artist list
        )

    def _parse_track(self, data: dict) -> Track:
        clean_data = self._clean_dict(data)
        
        # compute download_url without extra network
        file_path = clean_data.get("file")
        download_url = None
        if file_path and self.api_base_url:
            if file_path.startswith("http"):
                download_url = file_path
            elif file_path.startswith("/"):
                download_url = f"{self.api_base_url}{file_path}"
            else:
                download_url = f"{self.api_base_url}/{file_path}"

        credits = [TrackCredit(**self._clean_dict(c)) for c in clean_data.get("credits", [])]
        album_data = clean_data.get("album")
        album = self._parse_album(album_data) if album_data else None

        return Track(
            id=clean_data["id"],
            enName=clean_data.get("enName"),
            heName=clean_data.get("heName"),
            fileName=clean_data.get("fileName", ""),
            duration=clean_data.get("duration"),
            enLyrics=clean_data.get("enLyrics"),
            heLyrics=clean_data.get("heLyrics"),
            enDesc=clean_data.get("enDesc"),
            heDesc=clean_data.get("heDesc"),
            credits=credits,
            album=album,
            trackNumber=clean_data.get("trackNumber"),
            download_url=download_url,
        )

    def _parse_playlist(self, data: dict) -> Playlist:
        clean_data = self._clean_dict(data)
        return Playlist(
            id=clean_data["id"],
            name=clean_data.get("name", ""),
            enName=clean_data.get("enName", ""),
            heName=clean_data.get("heName", ""),
            image=clean_data.get("image"),
            private=clean_data.get("private"),
            featured=clean_data.get("featured"),
            premium=clean_data.get("premium"),
            user=clean_data.get("user")
        )


    # ======================
    # 🔍 Search
    # ======================

    def search_artists(self, query: str, size: int = 10) -> List[Artist]:
        elastic_body = {
            "query": {
                "bool": {
                    "must": [
                        {
                            "multi_match": {
                                "query": query,
                                "type": "most_fields",
                                "fuzziness": "AUTO",
                                "fields": ["enName", "heName"]
                            }
                        }
                    ]
                }
            },
            "from": 0,
            "size": size
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "index": "artists",
                "query": json.dumps(elastic_body)
            },
            "query": """query SearchElastic($index: String!, $query: String!) {
                __typename
                searchElastic(index: $index, query: $query)
            }"""
        }

        data = self._post(gql_payload)["searchElastic"]
        hits = json.loads(data).get("hits", {}).get("hits", [])

        results = []
        for hit in hits:
            source = hit["_source"]
            thumbnail = source.get("thumbnail")
            artist = Artist(
                id=source["id"],
                enName=source["enName"],
                heName=source["heName"],
                images=ImageSet(small=thumbnail) if thumbnail else None
            )
            results.append(artist)
        return results

    def search_tracks(self, query: str, size: int = 10) -> List[Track]:
        elastic_body = {
            "query": {
                "bool": {
                    "must": [
                        {
                            "multi_match": {
                                "query": query,
                                "type": "most_fields",
                                "fuzziness": "AUTO",
                                "fields": ["enName", "heName"]
                            }
                        }
                    ]
                }
            },
            "from": 0,
            "size": size
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "index": "tracks",
                "query": json.dumps(elastic_body)
            },
            "query": """query SearchElastic($index: String!, $query: String!) {
                __typename
                searchElastic(index: $index, query: $query)
            }"""
        }

        data = self._post(gql_payload)["searchElastic"]
        hits = json.loads(data).get("hits", {}).get("hits", [])

        results = []
        for hit in hits:
            source = hit["_source"]
            # Create a minimal track object from search results
            # We'll need to fetch full details if the track is played
            track = Track(
                id=source["id"],
                enName=source.get("enName", ""),
                heName=source.get("heName", ""),
                fileName=source.get("fileName", ""),
                duration=source.get("duration"),
                enLyrics=None,
                heLyrics=None,
                enDesc=None,
                heDesc=None,
                credits=[],
                album=None,  # We'll populate this from the search result
                trackNumber=source.get("trackNumber"),
                download_url=None  # Will be computed when needed
            )
            
            # Try to populate album info from search result if available
            if "album" in source and source["album"]:
                album_data = source["album"]
                track.album = Album(
                    id=album_data.get("id", 0),
                    enName=album_data.get("enName", ""),
                    heName=album_data.get("heName", ""),
                    releasedAt=album_data.get("releasedAt"),
                    images=self._parse_images(album_data.get("images")),
                    premium=album_data.get("premium"),
                    albumType=album_data.get("albumType"),
                    rssUrl=album_data.get("rssUrl"),
                    artists=[self._parse_artist(a) for a in album_data.get("artists", [])],
                    featuredArtists=[self._parse_artist(a) for a in album_data.get("featuredArtists", [])],
                )
            
            # If no album artists, try to get artist info from track level or other fields
            if track.album and (not track.album.artists and not track.album.featuredArtists):
                # Try artist fields directly on track
                if "artist" in source:
                    track.album.artists = [self._parse_artist(source["artist"])]
                elif "artists" in source and source["artists"]:
                    track.album.artists = [self._parse_artist(a) for a in source["artists"]]
                elif "artistName" in source and source["artistName"]:
                    # Create a minimal artist from name
                    track.album.artists = [Artist(
                        id=0,
                        enName=source["artistName"],
                        heName=source.get("artistHebrewName", source["artistName"]),
                        enDesc=None,
                        heDesc=None,
                        images=None
                    )]
            
            results.append(track)
        return results

    def search_podcasts(self, query: str, size: int = 10) -> List[Album]:
        """
        Search podcast 'albums' (feeds) by text in enName/heName.
        Filters to podcasts using (albumType == 'RSS') OR (rssUrl exists).
        Returns a list of Album objects (no tracks).
        """
        # Elasticsearch query: text match + podcast-only filter
        elastic_body = {
            "query": {
                "bool": {
                    "must": [
                        {
                            "multi_match": {
                                "query": query,
                                "type": "most_fields",
                                "fuzziness": "AUTO",
                                "fields": ["enName", "heName"]
                            }
                        }
                    ],
                    "should": [
                        {"term": {"albumType": "RSS"}},          # if albumType is indexed as a keyword/term
                        {"exists": {"field": "rssUrl"}},          # many podcast feeds have rssUrl
                    ],
                    "minimum_should_match": 1
                }
            },
            "from": 0,
            "size": size
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "index": "albums",
                "query": json.dumps(elastic_body)
            },
            "query": """
            query SearchElastic($index: String!, $query: String!) {
              __typename
              searchElastic(index: $index, query: $query)
            }"""
        }

        data = self._post(gql_payload)["searchElastic"]
        hits = json.loads(data).get("hits", {}).get("hits", [])

        results: List[Album] = []
        for hit in hits:
            src = hit.get("_source", {})

            # Extra safety: enforce podcast filter client-side too
            is_podcast = (src.get("albumType") == "RSS") or bool(src.get("rssUrl"))
            if not is_podcast:
                continue

            album = Album(
                id=src["id"],
                enName=src.get("enName", ""),
                heName=src.get("heName", ""),
                releasedAt=src.get("releasedAt"),
                images=self._parse_images(src.get("images")),
                premium=src.get("premium"),
                albumType=src.get("albumType"),
                rssUrl=src.get("rssUrl"),
                artists=[self._parse_artist(a) for a in src.get("artists", [])] if src.get("artists") else None,
                featuredArtists=[self._parse_artist(a) for a in src.get("featuredArtists", [])] if src.get("featuredArtists") else None,
            )
            results.append(album)

        return results

    def search_kids_albums(self, query: str, size: int = 10, include_podcasts: bool = False) -> List[Album]:
        """
        Search Kids albums using GraphQL 'albums' with a proper where filter.
        Heuristics:
        - genres contains 'kids' (English) or 'ילדים' (Hebrew)
        By default excludes podcasts (albumType == 'RSS').
        """
        where_or = [
            {"genres": {"some": {"enName": {"contains": "kids", "mode": "insensitive"}}}},
            {"genres": {"some": {"heName": {"contains": "ילדים", "mode": "insensitive"}}}},
        ]

        where: dict = {"OR": where_or}
        if not include_podcasts:
            where = {"AND": [where, {"albumType": {"not": {"equals": "RSS"}}}]}

        # Add text search if query is provided
        if query:
            name_clause = {
                "OR": [
                    {"enName": {"contains": query, "mode": "insensitive"}},
                    {"heName": {"contains": query, "mode": "insensitive"}},
                ]
            }
            if "AND" in where:
                where["AND"].append(name_clause)
            else:
                where = {"AND": [where, name_clause]}

        gql_payload = {
            "operationName": "SearchKidsAlbums",
            "variables": {
                "count": max(1, min(200, int(size or 10))),
                "skip": 0,
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "where": where,
            },
            "query": """
            query SearchKidsAlbums($count: Int!, $skip: Int!, $orderBy: [AlbumOrderByWithRelationInput!], $where: AlbumWhereInput!) {
              albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id enName heName releasedAt premium albumType rssUrl
                images { small medium large }
                artists { id enName heName }
                featuredArtists { id enName heName }
              }
            }
            """
        }

        data = self._post(gql_payload).get("albums", [])
        return [self._parse_album(a) for a in data]

    def search_kids_tracks(self, query: str, size: int = 10, include_podcasts: bool = False) -> List[Track]:
        """
        Search Kids tracks using GraphQL 'tracks' with proper where filter.
        Filters tracks that belong to albums with kids genres.
        Heuristics:
        - album.genres contains 'kids' (English) or 'ילדים' (Hebrew)
        By default excludes podcast tracks (album.albumType == 'RSS').
        """
        where_or = [
            {"album": {"genres": {"some": {"enName": {"contains": "kids", "mode": "insensitive"}}}}},
            {"album": {"genres": {"some": {"heName": {"contains": "ילדים", "mode": "insensitive"}}}}},
        ]

        where: dict = {"OR": where_or}
        if not include_podcasts:
            where = {"AND": [where, {"album": {"albumType": {"not": {"equals": "RSS"}}}}]}

        # Add text search if query is provided
        if query:
            name_clause = {
                "OR": [
                    {"enName": {"contains": query, "mode": "insensitive"}},
                    {"heName": {"contains": query, "mode": "insensitive"}},
                ]
            }
            if "AND" in where:
                where["AND"].append(name_clause)
            else:
                where = {"AND": [where, name_clause]}

        gql_payload = {
            "operationName": "SearchKidsTracks",
            "variables": {
                "count": max(1, min(200, int(size or 10))),
                "skip": 0,
                "orderBy": [{"album": {"releasedAt": "desc"}}, {"trackNumber": "asc"}, {"id": "desc"}],
                "where": where,
            },
            "query": """
            query SearchKidsTracks($count: Int!, $skip: Int!, $orderBy: [TrackOrderByWithRelationInput!], $where: TrackWhereInput!) {
              tracks(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id trackNumber enName heName fileName duration
                enLyrics heLyrics enDesc heDesc
                album {
                  id enName heName releasedAt premium albumType rssUrl
                  images { small medium large }
                  artists { id enName heName }
                  featuredArtists { id enName heName }
                }
              }
            }
            """
        }

        data = self._post(gql_payload).get("tracks", [])
        return [self._parse_track(t) for t in data]

    def search_playlists(self, query: str, size: int = 10) -> List[Playlist]:
        """Search for playlists by name."""
        gql_payload = {
            "operationName": "SearchPlaylists",
            "variables": {
                "where": {
                    "private": {"equals": False},
                    "playlistTracks": {"some": {}},
                    "OR": [
                        {"name": {"contains": query, "mode": "insensitive"}},
                        {"enName": {"contains": query, "mode": "insensitive"}},
                        {"heName": {"contains": query, "mode": "insensitive"}},
                    ],
                },
                "skip": 0,
                "count": max(1, min(100, int(size or 10))),
            },
            "query": """
            query SearchPlaylists($where: PlaylistWhereInput, $skip: Int, $count: Int) {
            playlists(where: $where, orderBy: {name: {sort: asc}}, take: $count, skip: $skip) {
                id
                name
                enName
                heName
                image
                private
                featured
                premium
                user {
                id
                uid
                name
                email
                }
            }
            }
            """
        }

        data = self._post(gql_payload)
        playlists_data = data.get("playlists", [])
        return [self._parse_playlist(playlist_data) for playlist_data in playlists_data]

    def search_albums(self, query: str, size: int = 10) -> List[Album]:
        """
        Search albums by name (English/Hebrew).
        """
        elastic_body = {
            "query": {
                "bool": {
                    "must": [
                        {
                            "multi_match": {
                                "query": query,
                                "type": "most_fields",
                                "fuzziness": "AUTO",
                                "fields": ["enName", "heName"]
                            }
                        }
                    ]
                }
            },
            "from": 0,
            "size": size
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "index": "albums",
                "query": json.dumps(elastic_body)
            },
            "query": """query SearchElastic($index: String!, $query: String!) {
                __typename
                searchElastic(index: $index, query: $query)
            }"""
        }

        data = self._post(gql_payload)["searchElastic"]
        hits = json.loads(data).get("hits", {}).get("hits", [])

        results: List["Album"] = []
        for hit in hits:
            src = hit["_source"]
            results.append(
                Album(
                    id=src.get("id"),
                    enName=src.get("enName"),
                    heName=src.get("heName"),
                    images=src.get("images"),
                    releasedAt=src.get("releasedAt"),
                    premium=src.get("premium"),
                )
            )
        return results

    # ======================
    # 👥 Get All Artists
    # ======================

    def get_all_artists(self, size: int = 1000, skip: int = 0) -> List[Artist]:
        elastic_body = {
            "query": {
                "match_all": {}
            },
            "from": skip,
            "size": size,
            "sort": [{"_score": "desc"}]
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "index": "artists",
                "query": json.dumps(elastic_body)
            },
            "query": """query SearchElastic($index: String!, $query: String!) {
                __typename
                searchElastic(index: $index, query: $query)
            }"""
        }

        data = self._post(gql_payload)["searchElastic"]
        hits = json.loads(data).get("hits", {}).get("hits", [])

        results = []
        for hit in hits:
            source = hit["_source"]
            thumbnail = source.get("thumbnail")
            artist = Artist(
                id=source["id"],
                enName=source["enName"],
                heName=source["heName"],
                images=ImageSet(small=thumbnail) if thumbnail else None
            )
            results.append(artist)
        return results



    # ======================
    # 🎵 Featured Playlists
    # ======================

    def get_featured_playlists(self, skip: int = 0, count: int = 10, genre_id: int = 7) -> List[Playlist]:
        gql_payload = {
            "operationName": "GetFeaturedPlaylist",
            "variables": {
                "where": {
                    "private": {"equals": False},
                    "featured": {"equals": True},
                    "playlistTracks": {"some": {}},
                    "OR": [
                        {"name": {"contains": "", "mode": "insensitive"}},
                        {"name": {"contains": "", "mode": "insensitive"}},
                    ],
                    "playlistGenres": {"some": {"id": {"equals": genre_id}}},
                },
                "skip": skip,
                "count": count,
            },
            "query": """
            query GetFeaturedPlaylist($where: PlaylistWhereInput, $skip: Int, $count: Int) {
            playlists(where: $where, orderBy: {index: {sort: asc}}, take: $count, skip: $skip) {
                id
                name
                enName
                heName
                image
                private
                featured
                premium
                user {
                id
                uid
                name
                email
                }
            }
            }
            """
        }

        data = self._post(gql_payload)  # assumes your API wrapper has _post() for GraphQL
        playlists_data = data.get("playlists", [])
        return [self._parse_playlist(playlist_data) for playlist_data in playlists_data]

    def get_artists_playlist(self, skip: int = 0, count: int = 10) -> List[Playlist]:
        """Get artists playlists (genre_id = 6)."""
        gql_payload = {
            "operationName": "GetArtistsPlaylist",
            "variables": {
                "where": {
                    "private": {"equals": False},
                    "featured": {"equals": True},
                    "playlistTracks": {"some": {}},
                    "OR": [
                        {"name": {"contains": "", "mode": "insensitive"}},
                        {"name": {"contains": "", "mode": "insensitive"}},
                    ],
                    "playlistGenres": {"some": {"id": {"equals": 6}}},
                },
                "skip": skip,
                "count": count,
            },
            "query": """
            query GetArtistsPlaylist($where: PlaylistWhereInput, $skip: Int, $count: Int) {
              playlists(where: $where, orderBy: {index: {sort: asc}}, take: $count, skip: $skip) {
                id
                name
                enName
                heName
                image
                private
                featured
                premium
                user {
                  id
                  uid
                  name
                  email
                }
              }
            }
            """,
        }

        data = self._post(gql_payload)
        playlists_data = data.get("playlists", [])
        return [self._parse_playlist(playlist_data) for playlist_data in playlists_data]

    def get_playlist_tracks(self, playlist_id: int) -> List[Track]:
        """Get tracks in a specific playlist."""
        gql_payload = {
            "operationName": "GetPlaylistTracks",
            "variables": {
                "playlistId": playlist_id
            },
            "query": """
            query GetPlaylistTracks($playlistId: Int!) {
                playlist(where: {id: $playlistId}) {
                    id
                    name
                    enName
                    heName
                    playlistTracks {
                        id
                        track {
                            id
                            enName
                            heName
                            fileName
                            duration
                            trackNumber
                            enLyrics
                            heLyrics
                            credits {
                                id
                                enName
                                heName
                                role
                            }
                            album {
                                id
                                enName
                                heName
                                images { small medium large }
                                artists {
                                    id
                                    enName
                                    heName
                                }
                            }
                        }
                    }
                }
            }
            """
        }

        data = self._post(gql_payload)
        playlist_data = data.get("playlist")
        if not playlist_data or not playlist_data.get("playlistTracks"):
            return []

        playlist_tracks = playlist_data["playlistTracks"]
        
        # Sort by playlistTrack ID to ensure consistent ordering
        # The playlistTrack ID represents the order in which tracks were added
        playlist_tracks.sort(key=lambda pt: pt.get('id', 0))

        tracks = []
        for playlist_track in playlist_tracks:
            track_data = playlist_track["track"]
            track = self._parse_track(track_data)
            tracks.append(track)

        return tracks

    def get_playlist_info(self, playlist_id: int) -> Optional[Playlist]:
        """Get playlist information."""
        gql_payload = {
            "operationName": "GetPlaylistInfo",
            "variables": {
                "playlistId": playlist_id
            },
            "query": """
            query GetPlaylistInfo($playlistId: Int!) {
                playlist(where: {id: $playlistId}) {
                    id
                    name
                    enName
                    heName
                    image
                    private
                    featured
                    premium
                    user {
                        id
                        uid
                        name
                        email
                    }
                }
            }
            """
        }

        data = self._post(gql_payload)
        playlist_data = data.get("playlist")
        return self._parse_playlist(playlist_data) if playlist_data else None


    # ======================
    # 🎵 Albums by Artist
    # ======================

    def get_artist_albums(self, artist_id: int, limit: int = 30, skip: int = 0) -> Tuple[Artist, List[Album], List[Album]]:
        gql_payload = {
            "operationName": None,
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 30))),
                "artistId": artist_id,
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "countWhere": {"artists": {"some": {"id": {"equals": artist_id}}}},
                "where": {
                    "artists": {"some": {"id": {"equals": artist_id}}},
                    "genres": {"none": {"enName": {"contains": "singles", "mode": "insensitive"}}}
                },
                "featuredWhere": {"featuredArtists": {"some": {"id": {"equals": artist_id}}}}
            },
            "query": """query GetArtistAlbums($skip: Int!, $count: Int!, $artistId: Int!, $countWhere: AlbumWhereInput!, $where: AlbumWhereInput!, $featuredWhere: AlbumWhereInput!, $orderBy: [AlbumOrderByWithRelationInput!]) {
            __typename
            albumsCount(where: $countWhere)
            artist(where: {id: $artistId}) {
                id
                enName
                heName
                images { small medium large }
                featuredAlbums(where: $featuredWhere) {
                id
                enName
                heName
                releasedAt
                images { small }
                }
            }
            albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id
                enName
                heName
                releasedAt
                images { small }
            }
            }"""
        }

        data = self._post(gql_payload)
        
        artist_data = data["artist"]
        artist = Artist(
            id=artist_data["id"],
            enName=artist_data["enName"],
            heName=artist_data["heName"],
            images=self._parse_images(artist_data.get("images"))
        )

        featured_albums = [self._parse_album(a) for a in artist_data.get("featuredAlbums", [])]
        albums = [self._parse_album(a) for a in data.get("albums", [])]

        return artist, albums, featured_albums
    # ======================
    # 🎶 Album Tracks
    # ======================

    def get_album_tracks(self, album_id: int, sort_order: str = "asc", limit: Optional[int] = None, skip: int = 0) -> Album:
        gql_payload = {
            "operationName": None,
            "variables": {
                "id": album_id,
                "sortOrder": sort_order,
                "featuredWhere": {},
                "count": limit,
                "skip": skip
            },
            "query": """query GetAlbumDetail($id: Int!, $sortOrder: SortOrder!, $featuredWhere: ArtistWhereInput, $count: Int, $skip: Int) {
                __typename
                album(where: {id: $id}) {
                    id enName heName releasedAt albumType rssUrl premium
                    images { small medium large }
                    artists { id enName heName }
                    featuredArtists(where: $featuredWhere) { id enName heName }
                    tracks(orderBy: {trackNumber: $sortOrder}, take: $count, skip: $skip) {
                        id trackNumber enName heName fileName duration file
                        enLyrics heLyrics
                        credits { id enName heName role }
                    }
                }
            }"""
        }

        response = self._post(gql_payload)
        
        if not response or "album" not in response:
            print(f"[ZING API] Invalid response for album {album_id}: {response}")
            return None
            
        data = response["album"]
        
        if not data:
            print(f"[ZING API] Album {album_id} not found or unavailable")
            return None

        album = Album(
            id=data["id"],
            enName=data["enName"],
            heName=data["heName"],
            releasedAt=data["releasedAt"],
            images=self._parse_images(data.get("images")),
            premium=data.get("premium"),
            albumType=data.get("albumType"),
            rssUrl=data.get("rssUrl"),
            artists=[self._parse_artist(a) for a in data.get("artists", [])],
            featuredArtists=[self._parse_artist(a) for a in data.get("featuredArtists", [])],
        )

        # album.tracks = [self._parse_track(t) for t in data.get("tracks", [])]
        album.tracks = []
        for t in data.get("tracks", []):
            track = self._parse_track(t)
            track.album = album  # ✅ inject the album object manually
            album.tracks.append(track)

        track_numbers = [t.trackNumber for t in album.tracks if t.trackNumber is not None]
        if track_numbers:
            last_track_number = max(track_numbers)
            for track in album.tracks:
                track.is_last_track = (track.trackNumber == last_track_number)
        else:
            for track in album.tracks:
                track.is_last_track = False

        self._album_cache[album_id] = album
        return album


    def get_artist_singles(
        self,
        artist_id: int,
        limit: int = 10,
        skip: int = 0,
        genre_name: str = "singles",
    ):

        take = max(1, min(200, int(limit or 10)))
        offset = max(0, int(skip or 0))

        gql_payload = {
            "operationName": "GetArtistSingles",
            "variables": {
                "skip": offset,
                "count": take,
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "where": {
                    "genres": {
                        "some": {
                            "enName": {
                                "contains": str(genre_name or "singles"),
                                "mode": "insensitive",
                            }
                        }
                    },
                    "artists": {"some": {"id": {"equals": int(artist_id)}}},
                },
            },
            "query": """
            query GetArtistSingles($count: Int!, $skip: Int!, $orderBy: [AlbumOrderByWithRelationInput!], $where: AlbumWhereInput!) {
            __typename
            albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                enName
                heName
                releasedAt
                label {
                __typename
                id
                enName
                heName
                }
                images {
                __typename
                small
                medium
                large
                }
                artists {
                __typename
                enName
                heName
                }
                premium
                albumType
                rssUrl
                tracks {
                __typename
                id
                }
            }
            }
            """,
        }

        data = self._post(gql_payload)
        # Expecting: {"albums": [ ... ]}
        return data.get("albums", [])


    def get_artist_songs(
        self,
        artist_id: int,
        limit: int = 200,
        skip: int = 0,
        fetch_all: bool = False,
    ) -> Tuple[List["Track"], int]:
        """
        List songs (tracks) by artist ID.

        Returns:
            (tracks, total_count)
            - tracks: List[Track] (or raw dicts if Track parser is unavailable)
            - total_count: total number of matching tracks on the server

        Args:
            artist_id: The artist ID to filter by.
            limit: Page size (1..200). Ignored when fetch_all=True (internally pages with 200).
            skip: Offset for pagination. Ignored when fetch_all=True (starts from 0).
            fetch_all: If True, fetches all tracks across pages.
        """
        # Normalize paging
        page_size = 200 if fetch_all else max(1, min(200, int(limit or 30)))
        cursor = 0 if fetch_all else max(0, int(skip or 0))

        def _build_payload(take: int, offset: int) -> dict:
            return {
                "operationName": "GetTracksByArtist",
                "variables": {
                    "artistId": int(artist_id),
                    "skip": int(offset),
                    "count": int(take),
                },
                "query": """
                query GetTracksByArtist($artistId: Int!, $skip: Int!, $count: Int!) {
                __typename
                tracksCount(where: { artists: { some: { id: { equals: $artistId } } } })
                tracks(
                    take: $count
                    skip: $skip
                    orderBy: [{ album: { releasedAt: desc } }, { id: desc }]
                    where: { artists: { some: { id: { equals: $artistId } } } }
                ) {
                    __typename
                    id
                    enName
                    heName
                    fileName
                    enLyrics
                    heLyrics
                    enDesc
                    heDesc
                    duration
                    credits {
                    __typename
                    id
                    enName
                    heName
                    role
                    }
                    album {
                    __typename
                    id
                    enName
                    heName
                    releasedAt
                    albumType
                    rssUrl
                    premium
                    images {
                        __typename
                        small
                        medium
                        large
                    }
                    artists {
                        __typename
                        id
                        enName
                        heName
                        images {
                        __typename
                        small
                        medium
                        large
                        }
                    }
                    }
                }
                }
                """,
            }

        # Optional Track parser if your class provides one
        parse_track = getattr(self, "_track_from_gql", None)
        to_track = (lambda t: parse_track(t)) if callable(parse_track) else (lambda t: t)

        all_items: List[Any] = []
        total_count: Optional[int] = None

        while True:
            payload = _build_payload(page_size, cursor)
            try:
                resp = self._post(payload)
            except Exception as e:
                # Keep your logger style consistent
                try:
                    _LOGGER.error("get_artist_songs failed: %s", e)
                except Exception:
                    pass
                raise

            if total_count is None:
                total_count = int(resp.get("tracksCount", 0))

            items = resp.get("tracks") or []
            if not items:
                break

            all_items.extend(to_track(t) for t in items)

            if not fetch_all:
                break

            cursor += len(items)
            if cursor >= total_count:
                break

        return all_items, int(total_count or 0)



    def get_artist_latest_releases(
        self,
        artist_id: int,
        limit: int = 10,
        skip: int = 0,
        fetch_all: bool = False,
    ) -> Tuple[List["Album"], int]:
        """
        Get latest releases (albums) by artist ID, ordered by release date (desc).

        Returns:
            (albums, total_count)
            - albums: List[Album] (or raw dicts if Album parser is unavailable)
            - total_count: total number of matching albums on the server

        Args:
            artist_id: The artist ID to filter by.
            limit: Page size (1..200). Ignored when fetch_all=True (internally uses 200).
            skip: Offset for pagination. Ignored when fetch_all=True (starts from 0).
            fetch_all: If True, fetches all albums across pages.
        """
        # Normalize paging
        page_size = 200 if fetch_all else max(1, min(200, int(limit or 30)))
        cursor = 0 if fetch_all else max(0, int(skip or 0))

        def _build_payload(take: int, offset: int) -> dict:
            return {
                "operationName": "GetLatestReleasesByArtist",
                "variables": {
                    "artistId": int(artist_id),
                    "skip": int(offset),
                    "count": int(take),
                },
                "query": """
                query GetLatestReleasesByArtist($artistId: Int!, $skip: Int!, $count: Int!) {
                __typename
                albumsCount(where: { artists: { some: { id: { equals: $artistId } } } })
                albums(
                    take: $count
                    skip: $skip
                    orderBy: [{ releasedAt: desc }, { id: desc }]
                    where: { artists: { some: { id: { equals: $artistId } } } }
                ) {
                    __typename
                    id
                    enName
                    heName
                    releasedAt
                    albumType
                    rssUrl
                    premium
                    label {
                    __typename
                    id
                    enName
                    heName
                    }
                    images {
                    __typename
                    small
                    medium
                    large
                    }
                    artists {
                    __typename
                    id
                    enName
                    heName
                    images {
                        __typename
                        small
                        medium
                        large
                    }
                    }
                    tracks {
                    __typename
                    id
                    enName
                    heName
                    duration
                    }
                }
                }
                """,
            }

        # Optional Album parser if your class provides one
        parse_album = getattr(self, "_album_from_gql", None)
        to_album = (lambda a: parse_album(a)) if callable(parse_album) else (lambda a: a)

        all_items: List[Any] = []
        total_count: Optional[int] = None

        while True:
            payload = _build_payload(page_size, cursor)
            try:
                resp = self._post(payload)
            except Exception as e:
                try:
                    _LOGGER.error("get_artist_latest_releases failed: %s", e)
                except Exception:
                    pass
                raise

            if total_count is None:
                total_count = int(resp.get("albumsCount", 0))

            items = resp.get("albums") or []
            if not items:
                break

            all_items.extend(to_album(a) for a in items)

            if not fetch_all:
                break

            cursor += len(items)
            if cursor >= total_count:
                break

        return all_items, int(total_count or 0)

    # ======================
    # 🔗 Track Download URL
    # ======================

    def get_track_download_url(self, track_id: int) -> Optional[str]:
        gql_payload = {
            "operationName": "GetTrackById",
            "variables": {"id": track_id},
            "query": """
            query GetTrackById($id: Int!) {
              track(where: { id: $id }) {
                file
              }
            }
            """
        }

        track_data = self._post(gql_payload).get("track")
        if not track_data or not track_data.get("file"):
            return None

        file_path = track_data["file"]
        if file_path.startswith("http"):
            return file_path
        elif self.api_base_url:
            if file_path.startswith("/"):
                return f"{self.api_base_url}{file_path}"
            else:
                return f"{self.api_base_url}/{file_path}"
        return None

    # ======================
    # 🔍 Get Track by ID
    # ======================

    def get_track_by_id(self, track_id: int) -> Optional[Track]:
        gql_payload = gql_payload = {
            "operationName": "GetTrackById",
            "variables": {"id": track_id},
            "query": """
            query GetTrackById($id: Int!) {
            track(where: { id: $id }) {
                id
                trackNumber
                enName
                heName
                fileName
                duration
                enLyrics
                heLyrics
                enDesc
                heDesc
                file
                credits {
                id
                enName
                heName
                role
                }
                album {
                id
                enName
                heName
                releasedAt
                albumType
                rssUrl
                premium
                images {
                    small
                    medium
                    large
                }
                artists {
                    id
                    enName
                    heName
                    images {
                    small
                    medium
                    large
                    }
                }
                featuredArtists {
                    id
                    enName
                    heName
                    images {
                    small
                    medium
                    large
                    }
                }
                }
            }
            }
            """
        }

        data = self._post(gql_payload).get("track")
        if not data:
            return None
        return self._parse_track(data)

    # ======================
    # 🕒 Recent Tracks
    # ======================
    def get_recent_tracks(self, limit: int = 100) -> List[Track]:
        """Return the most recently added tracks (descending by id as a stable proxy).
        Also includes minimal album + artists + images for UI thumbnails.
        """
        gql_payload = {
            "operationName": None,
            "variables": {
                "count": max(1, min(200, int(limit or 100))),
            },
            "query": """
            query GetRecentTracks($count: Int!) {
              __typename
              tracks(take: $count, orderBy: { id: desc }) {
                id
                trackNumber
                enName
                heName
                fileName
                duration
                file
                album {
                  id
                  enName
                  heName
                  images { small medium large }
                  artists { id enName heName }
                }
              }
            }
            """,
        }

        try:
            data = self._post(gql_payload).get("tracks", [])
        except Exception:
            data = []
        results: List[Track] = []
        for t in data:
            try:
                results.append(self._parse_track(t))
            except Exception:
                continue
        return results

    # ======================
    # 🎙️ Podcasts
    # ======================

    def get_podcast_albums(self, limit: int = 30, skip: int = 0) -> List[Album]:
        """
        Return podcast 'albums' (feeds) by filtering albums where albumType == 'RSS'.
        You can change the orderBy as needed (releasedAt desc then id desc is a good default).
        """
        gql_payload = {
            "operationName": "GetPodcastAlbums",
            "variables": {
                "count": max(1, min(200, int(limit or 30))),
                "skip": max(0, int(skip or 0)),
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "where": {"albumType": {"equals": "RSS"}},
            },
            "query": """
            query GetPodcastAlbums($count: Int!, $skip: Int!, $orderBy: [AlbumOrderByWithRelationInput!], $where: AlbumWhereInput!) {
              albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id
                enName
                heName
                releasedAt
                images { small medium large }
                premium
                albumType
                rssUrl
                artists { id enName heName }
                featuredArtists { id enName heName }
              }
            }
            """
        }

        try:
            data = self._post(gql_payload).get("albums", [])
        except Exception:
            data = []
        
        results: List[Album] = []
        for album_data in data:
            try:
                results.append(self._parse_album(album_data))
            except Exception:
                continue
        return results

    def get_podcasts_hosts(
        self,
        limit: int = 10,
        skip: int = 0,
        search: str = "",
        order_by: Optional[List[Dict[str, str]]] = None,
    ) -> List["Artist"]:
        """
        Return podcast hosts (artists tagged with category 'podcasts').

        Args:
            limit: Max number of results to return (1..200).
            skip: Offset for pagination.
            search: Optional text to match in enName/heName (case-insensitive).
            order_by: Optional GraphQL order array, e.g. [{"enName": "asc"}].
                      Defaults to enName ASC.

        Returns:
            List[Artist]
        """
        order_by = order_by or [{"enName": "asc"}]

        # Build the name filter. Empty string behaves like "match all" (as in your sample).
        name_filter = {
            "OR": [
                {"enName": {"contains": search or "", "mode": "insensitive"}},
                {"heName": {"contains": search or "", "mode": "insensitive"}},
            ]
        }

        where = {
            **name_filter,
            "categories": {"some": {"enName": {"contains": "podcasts", "mode": "insensitive"}}},
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 10))),
                "orderBy": order_by,
                "where": where,
            },
            "query": """query GetCatArtists($skip: Int!, $count: Int!, $orderBy: [ArtistOrderByWithRelationInput!], $where: ArtistWhereInput!) {
              __typename
              artists(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                enName
                heName
                images {
                  __typename
                  small
                  medium
                  large
                }
              }
            }""",
        }

        data = self._post(gql_payload)
        items = data["artists"]

        # Map to Artist objects using the existing _parse_artist method
        results: List["Artist"] = []
        for artist_data in items:
            try:
                results.append(self._parse_artist(artist_data))
            except Exception:
                continue
        return results

    def get_artist_podcasts(
        self,
        artist_id: int,
        limit: int = 10,
        skip: int = 0,
        include_featured: bool = True,
    ) -> Tuple[Dict[str, Any], List[Dict[str, Any]], List[Dict[str, Any]], int]:
        """
        Return podcasts (RSS albums) for a given artist.

        Args:
            artist_id: The artist ID.
            limit: Max number of podcast albums to return (1..200).
            skip: Pagination offset.
            include_featured: If True, also fetch artist.featuredAlbums filtered to podcasts.

        Returns:
            (artist, featured_podcasts, podcasts, total_count)
            - artist: dict with id/names/descriptions/images/socials
            - featured_podcasts: list of featured podcast albums (may be empty if include_featured=False)
            - podcasts: list of podcast albums for this artist (paged)
            - total_count: total number of podcast albums for this artist
        """
        # GraphQL order: newest releases first, then stable by id desc
        order_by = [{"releasedAt": "desc"}, {"id": "desc"}]

        # Count only podcast albums for this artist
        count_where = {
            "artists": {"some": {"id": {"equals": artist_id}}},
            "albumType": {"equals": "RSS"},
        }

        # Page query: only podcast albums for this artist
        where = {
            "artists": {"some": {"id": {"equals": artist_id}}},
            "albumType": {"equals": "RSS"},
        }

        # Featured section: also filtered to podcasts
        featured_where = {
            "featuredArtists": {"some": {"id": {"equals": artist_id}}},
            "albumType": {"equals": "RSS"},
        }

        gql_payload = {
            "operationName": None,
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 10))),
                "artistId": int(artist_id),
                "orderBy": order_by,
                "countWhere": count_where,
                "where": where,
                "featuredWhere": featured_where,
            },
            "query": """query GetArtistPodcasts(
              $skip: Int!, $count: Int!, $artistId: Int!,
              $countWhere: AlbumWhereInput!, $where: AlbumWhereInput!,
              $featuredWhere: AlbumWhereInput!, $orderBy: [AlbumOrderByWithRelationInput!]
            ) {
              __typename
              albumsCount(where: $countWhere)
              artist(where: {id: $artistId}) {
                __typename
                id
                enName
                heName
                images {
                  __typename
                  small
                  medium
                  large
                  enCredit
                  heCredit
                }
                enDesc
                heDesc
                socials { __typename type url }
                featuredAlbums(where: $featuredWhere) @include(if: true) {
                  __typename
                  id
                  enName
                  heName
                  releasedAt
                  artists { __typename enName heName }
                  label { __typename id enName heName }
                  images { __typename small medium large }
                  premium
                  albumType
                  rssUrl
                }
              }
              albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                enName
                heName
                releasedAt
                label { __typename id enName heName }
                images { __typename small medium large }
                artists { __typename enName heName }
                premium
                albumType
                rssUrl
              }
            }"""
        }

        # If the caller doesn't want featured podcasts, strip that part from the query to save payload
        if not include_featured:
            gql_payload["query"] = gql_payload["query"].replace(
                "featuredAlbums(where: $featuredWhere) @include(if: true) {",
                "featuredAlbums(where: $featuredWhere) @include(if: false) {"
            )

        data = self._post(gql_payload)

        total_count = data.get("albumsCount", 0)
        artist = data.get("artist") or {}
        featured = artist.get("featuredAlbums") if include_featured else []
        podcasts = data.get("albums", [])

        # Map to proper objects using existing parsers
        parsed_featured = []
        for album_data in (featured or []):
            try:
                parsed_featured.append(self._parse_album(album_data))
            except Exception:
                continue
                
        parsed_podcasts = []
        for album_data in podcasts:
            try:
                parsed_podcasts.append(self._parse_album(album_data))
            except Exception:
                continue

        return artist, parsed_featured, parsed_podcasts, int(total_count)

    # ======================
    # 👶 Kids Content
    # ======================

    def get_kids_albums(self, limit: int = 30, skip: int = 0) -> List[Album]:
        gql_payload = {
            "operationName": "GetKidsAlbums",
            "variables": {
                "count": max(1, min(200, int(limit or 30))),
                "skip": max(0, int(skip or 0)),
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "where": {
                    "OR": [
                        {"genres": {"some": {"enName": {"contains": "kids", "mode": "insensitive"}}}},
                        {"genres": {"some": {"enName": {"contains": "children", "mode": "insensitive"}}}},
                        {"genres": {"some": {"heName": {"contains": "ילדים", "mode": "insensitive"}}}},
                        {"enName": {"contains": "kids", "mode": "insensitive"}},
                        {"enName": {"contains": "children", "mode": "insensitive"}},
                        {"heName": {"contains": "ילדים", "mode": "insensitive"}},
                    ]
                },
            },
            "query": """
            query GetKidsAlbums($count: Int!, $skip: Int!, $orderBy: [AlbumOrderByWithRelationInput!], $where: AlbumWhereInput!) {
              albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id
                enName
                heName
                releasedAt
                images { small medium large }
                premium
                albumType
                rssUrl
                artists { id enName heName }
                featuredArtists { id enName heName }
              }
            }
            """
        }

        try:
            data = self._post(gql_payload).get("albums", [])
        except Exception:
            data = []
        
        results: List[Album] = []
        for album_data in data:
            try:
                results.append(self._parse_album(album_data))
            except Exception:
                continue
        return results

    def get_kids_artists(self, limit: int = 30, skip: int = 0, order_by: Optional[list] = None) -> List["Artist"]:
        def _fetch(where_relation: str) -> List[dict]:
            gql_payload = {
                "operationName": "GetCatArtists",
                "variables": {
                    "skip": max(0, int(skip or 0)),
                    "count": max(1, min(200, int(limit or 30))),
                    "orderBy": order_by or [{"enName": "asc"}, {"id": "asc"}],
                    "where": {
                        where_relation: {
                            "some": {
                                "enName": {
                                    "contains": "kids",
                                    "mode": "insensitive"
                                }
                            }
                        }
                    }
                },
                "query": """
                query GetCatArtists($skip: Int!, $count: Int!, $orderBy: [ArtistOrderByWithRelationInput!], $where: ArtistWhereInput!) {
                __typename
                artists(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                    __typename
                    id
                    enName
                    heName
                    images {
                    __typename
                    small
                    medium
                    large
                    }
                }
                }
                """,
            }
            data = self._post(gql_payload)
            return data.get("artists", []) if isinstance(data, dict) else []

        # First try exactly your schema guess: 'categories'
        artists_json = _fetch("categories")

        # If empty, try a common alternative relation name
        if not artists_json:
            artists_json = _fetch("artistCategories")

        # Convert to Artist dataclass objects
        artists = []
        for artist_data in artists_json:
            try:
                artist = Artist(
                    id=artist_data["id"],
                    enName=artist_data["enName"],
                    heName=artist_data["heName"],
                    images=self._parse_images(artist_data.get("images"))
                )
                artists.append(artist)
            except Exception as e:
                _LOGGER.warning(f"Failed to parse kids artist {artist_data}: {e}")
        
        return artists

    def get_kids_artist_albums(self, artist_id: int, limit: int = 30, skip: int = 0) -> List[Album]:
        """Get kids albums for a specific artist."""
        gql_payload = {
            "operationName": "GetKidsArtistAlbums",
            "variables": {
                "count": max(1, min(200, int(limit or 30))),
                "skip": max(0, int(skip or 0)),
                "orderBy": [{"releasedAt": "desc"}, {"id": "desc"}],
                "where": {
                    "AND": [
                        {
                            "OR": [
                                {"artists": {"some": {"id": {"equals": artist_id}}}},
                                {"featuredArtists": {"some": {"id": {"equals": artist_id}}}}
                            ]
                        },
                        {
                            "OR": [
                                {"genres": {"some": {"enName": {"contains": "kids", "mode": "insensitive"}}}},
                                {"genres": {"some": {"enName": {"contains": "children", "mode": "insensitive"}}}},
                                {"genres": {"some": {"heName": {"contains": "ילדים", "mode": "insensitive"}}}},
                                {"enName": {"contains": "kids", "mode": "insensitive"}},
                                {"enName": {"contains": "children", "mode": "insensitive"}},
                                {"heName": {"contains": "ילדים", "mode": "insensitive"}},
                            ]
                        }
                    ]
                },
            },
            "query": """
            query GetKidsArtistAlbums($count: Int!, $skip: Int!, $orderBy: [AlbumOrderByWithRelationInput!], $where: AlbumWhereInput!) {
              albums(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                id
                enName
                heName
                releasedAt
                images { small medium large }
                premium
                albumType
                rssUrl
                artists { id enName heName }
                featuredArtists { id enName heName }
              }
            }
            """
        }

        try:
            data = self._post(gql_payload).get("albums", [])
        except Exception:
            data = []
        
        results: List[Album] = []
        for album_data in data:
            try:
                results.append(self._parse_album(album_data))
            except Exception:
                continue
        return results

    def get_kids_artist_latest(self, artist_id: int, limit: int = 20, skip: int = 0) -> List[Track]:
        """Get latest kids tracks for a specific artist."""
        gql_payload = {
            "operationName": "GetKidsArtistLatest",
            "variables": {
                "count": max(1, min(200, int(limit or 20))),
                "skip": max(0, int(skip or 0)),
                "orderBy": [{"album": {"releasedAt": "desc"}}, {"id": "desc"}],
                "where": {
                    "AND": [
                        {
                            "OR": [
                                {"album": {"artists": {"some": {"id": {"equals": artist_id}}}}},
                                {"album": {"featuredArtists": {"some": {"id": {"equals": artist_id}}}}}
                            ]
                        },
                        {
                            "album": {
                                "OR": [
                                    {"genres": {"some": {"enName": {"contains": "kids", "mode": "insensitive"}}}},
                                    {"genres": {"some": {"enName": {"contains": "children", "mode": "insensitive"}}}},
                                    {"genres": {"some": {"heName": {"contains": "ילדים", "mode": "insensitive"}}}},
                                    {"enName": {"contains": "kids", "mode": "insensitive"}},
                                    {"enName": {"contains": "children", "mode": "insensitive"}},
                                    {"heName": {"contains": "ילדים", "mode": "insensitive"}},
                                ]
                            }
                        }
                    ]
                },
            },
            "query": """
            query GetKidsArtistLatest($count: Int!, $skip: Int!, $orderBy: [TrackOrderByWithRelationInput!], $where: TrackWhereInput!) {
              tracks(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                trackNumber
                enName
                heName
                fileName
                duration
                file
                enLyrics
                heLyrics
                enDesc
                heDesc
                credits { __typename id enName heName role }
                album {
                  __typename
                  id
                  enName
                  heName
                  releasedAt
                  premium
                  albumType
                  rssUrl
                  images { __typename small medium large }
                  artists { __typename id enName heName }
                  featuredArtists { __typename id enName heName }
                }
              }
            }
            """
        }

        try:
            data = self._post(gql_payload).get("tracks", [])
        except Exception:
            data = []
        
        results: List[Track] = []
        for track_data in data:
            try:
                results.append(self._parse_track(track_data))
            except Exception:
                continue
        return results


    def get_latest_singles(self, limit: int = 10, skip: int = 0) -> List[Track]:
        gql_payload = {
            "operationName": "GetTracksByGenre",
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 10))),
                "orderBy": [
                    {"album": {"releasedAt": "desc"}},
                    {"id": "desc"},
                ],
                "where": {
                    "album": {
                        "genres": {"some": {"id": {"equals": 36}}}
                    }
                },
            },
            "query": """
            query GetTracksByGenre(
            $where: TrackWhereInput!,
            $skip: Int!,
            $count: Int!,
            $orderBy: [TrackOrderByWithRelationInput!]
            ) {
            __typename
            tracksCount(where: $where)
            tracks(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                enName
                heName
                fileName
                enLyrics
                heLyrics
                enDesc
                heDesc
                duration
                credits { __typename id enName heName role }
                album {
                __typename
                id
                enName
                heName
                images { __typename small medium large }
                premium
                albumType
                rssUrl
                }
            }
            }
            """,
        }

        resp = self._post(gql_payload)
        data = resp.get("data", resp)
        tracks_data = data.get("tracks", [])
        return [self._parse_track(track) for track in tracks_data]


    def get_latest_songs(self, limit: int = 10, skip: int = 0) -> List[Track]:
        payload = {
            "operationName": "GetTracksByGenre",
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 10))),
                "orderBy": [
                    {"album": {"releasedAt": "desc"}},
                    {"id": "desc"},
                ],
                "where": {
                    "album": {
                        "genres": {"some": {"id": {"equals": 34}}}
                    }
                },
            },
            "query": """
            query GetTracksByGenre(
            $where: TrackWhereInput!,
            $skip: Int!,
            $count: Int!,
            $orderBy: [TrackOrderByWithRelationInput!]
            ) {
            __typename
            tracksCount(where: $where)
            tracks(take: $count, skip: $skip, orderBy: $orderBy, where: $where) {
                __typename
                id
                enName
                heName
                fileName
                enLyrics
                heLyrics
                enDesc
                heDesc
                duration
                credits { __typename id enName heName role }
                album {
                __typename
                id
                enName
                heName
                images { __typename small medium large }
                premium
                albumType
                rssUrl
                }
            }
            }
            """,
        }
        resp = self._post(payload)
        data = resp.get("data", resp)
        tracks_data = data.get("tracks", [])
        return [self._parse_track(track) for track in tracks_data]


    def get_latest_albums(self, limit: int = 30, skip: int = 0) -> List[Album]:
        gql_payload = {
            "operationName": "GetGenreAlbums",
            "variables": {
                "skip": max(0, int(skip or 0)),
                "count": max(1, min(200, int(limit or 30))),
                "orderBy": [
                    {"releasedAt": "desc"},
                    {"id": "desc"}
                ],
                "where": {
                    "genres": {
                        "some": {
                            "id": {"equals": 34}
                        }
                    }
                }
            },
            "query": """
            query GetGenreAlbums(
            $skip: Int!,
            $count: Int!,
            $orderBy: [AlbumOrderByWithRelationInput!],
            $where: AlbumWhereInput!
            ) {
            __typename
            albumsCount(where: $where)
            albums(
                take: $count,
                skip: $skip,
                orderBy: $orderBy,
                where: $where
            ) {
                __typename
                id
                enName
                heName
                releasedAt
                label { __typename id enName heName }
                images { __typename small medium large }
                artists { __typename id enName heName }
                premium
                albumType
                rssUrl
                genres { __typename id enName heName }
            }
            }
            """
        }

        resp = self._post(gql_payload)
        data = resp.get("data", resp)
        albums_data = data.get("albums", [])
        return [self._parse_album(album) for album in albums_data]



class PlaybackController:
    def __init__(self, api: "ZingMusicAPI"):
        self.api = api
        self._queue: List[Track] = []
        self._current_index = 0
        self._shuffle = False
        self._repeat = "off"  # "off", "one", "all"
        self._original_queue: List[Track] = []  # Keep original order for shuffle toggle

    def set_queue(self, tracks: List[Track], start_index: int = 0, shuffle: bool = False) -> Optional[Track]:
        """Set the playback queue with tracks in consistent order."""
        if not tracks:
            return None
            
        self._original_queue = tracks.copy()
        self._queue = tracks.copy()
        self._current_index = start_index
        
        if shuffle:
            self.set_shuffle(True)
        
        return self.get_current_track()

    def play_album(self, album_id: int, start_track_id: Optional[int] = None, shuffle: bool = False) -> Optional["Track"]:
        """Load album tracks into queue and start playing."""
        try:
            album = self.api.get_album_tracks(album_id)
            tracks = album.tracks or []

            # Sort tracks by trackNumber if available, otherwise use list order
            if any(t.trackNumber is not None for t in tracks):
                # Keep only tracks with valid trackNumber and sort them
                tracks = [t for t in tracks if t.trackNumber is not None]
                tracks.sort(key=lambda t: t.trackNumber)
            
            start_index = 0
            if start_track_id:
                for i, track in enumerate(tracks):
                    if track.id == start_track_id:
                        start_index = i
                        break
                        
            return self.set_queue(tracks, start_index, shuffle)
        except Exception as e:
            print(f"[ZING] Error loading album {album_id}: {e}")
            return None

    def play_playlist(self, playlist_id: int, start_track_id: Optional[int] = None, shuffle: bool = False) -> Optional["Track"]:
        """Load playlist tracks into queue and start playing."""
        try:
            tracks = self.api.get_playlist_tracks(playlist_id)
            if not tracks:
                return None
                
            start_index = 0
            if start_track_id:
                for i, track in enumerate(tracks):
                    if track.id == start_track_id:
                        start_index = i
                        break
                        
            return self.set_queue(tracks, start_index, shuffle)
        except Exception as e:
            print(f"[ZING] Error loading playlist {playlist_id}: {e}")
            return None

    def play_next(self) -> Optional["Track"]:
        """Move to next track in queue."""
        if not self._queue:
            return None
            
        if self._repeat == "one":
            # Stay on current track
            return self.get_current_track()
        elif self._repeat == "all" and self._current_index >= len(self._queue) - 1:
            # Loop back to beginning
            self._current_index = 0
        else:
            # Normal next
            if self._current_index + 1 < len(self._queue):
                self._current_index += 1
            else:
                return None  # End of playlist
            
        current_track = self.get_current_track()
        if current_track:
            print(f"[ZING] Playing next track: {current_track.id} ({current_track.enName})")
        return current_track

    def play_previous(self) -> Optional["Track"]:
        """Move to previous track in queue."""
        if not self._queue:
            return None
            
        if self._repeat == "one":
            # Stay on current track
            return self.get_current_track()
        elif self._repeat == "all" and self._current_index <= 0:
            # Loop to end
            self._current_index = len(self._queue) - 1
        else:
            # Normal previous
            if self._current_index > 0:
                self._current_index -= 1
            else:
                return None  # Beginning of playlist
            
        return self.get_current_track()

    def get_current_track(self) -> Optional["Track"]:
        """Get currently playing track."""
        if not self._queue or self._current_index >= len(self._queue) or self._current_index < 0:
            return None
        return self._queue[self._current_index]

    def get_queue(self) -> List[Track]:
        """Get the current queue."""
        return self._queue.copy()

    def get_track_position(self) -> tuple[int, int]:
        """Get current position in queue."""
        return (self._current_index + 1, len(self._queue))

    def set_repeat(self, repeat: str = "off"):
        """Set repeat mode: off, one, all."""
        self._repeat = repeat

    def set_shuffle(self, shuffle: bool = True):
        """Enable/disable shuffle mode."""
        if shuffle == self._shuffle:
            return
            
        self._shuffle = shuffle
        
        if not self._queue:
            return
            
        current_track = self.get_current_track()
        
        if shuffle:
            # Shuffle the queue but keep current track at current position
            import random
            remaining_tracks = self._queue[self._current_index + 1:]
            random.shuffle(remaining_tracks)
            self._queue = self._queue[:self._current_index + 1] + remaining_tracks
        else:
            # Restore original order, find current track's new position
            self._queue = self._original_queue.copy()
            if current_track:
                for i, track in enumerate(self._queue):
                    if track.id == current_track.id:
                        self._current_index = i
                        break

    def has_next(self) -> bool:
        """Check if there's a next track available."""
        if not self._queue:
            return False
        if self._repeat in ["one", "all"]:
            return True
        return self._current_index < len(self._queue) - 1

    def has_previous(self) -> bool:
        """Check if there's a previous track available."""
        if not self._queue:
            return False
        if self._repeat in ["one", "all"]:
            return True
        return self._current_index > 0

    # Legacy properties for compatibility
    @property
    def playlist(self) -> List[Track]:
        return self._queue

    @property
    def current_index(self) -> int:
        return self._current_index