from __future__ import annotations

from homeassistant.core import HomeAssistant
from homeassistant.config_entries import ConfigEntry
from homeassistant.helpers.dispatcher import async_dispatcher_send
from homeassistant.exceptions import ConfigEntryAuthFailed, ConfigEntryNotReady
from homeassistant.const import CONF_EMAIL, CONF_PASSWORD
from .licenser_helpers import prepare_licensed_integration

from .const import DOMAIN, CONF_ALLOWED_ENTITY_IDS, SIGNAL_OPTIONS_UPDATED
from .api import ZingMusicAPI, ZingMusicUserError

PLATFORMS = ["media_player"]


async def async_setup_entry(hass: HomeAssistant, entry: ConfigEntry):
    # 1. License Check
    result = await prepare_licensed_integration(hass, entry, DOMAIN)
    if not result:
        return False
    license_id, remote_ns = result

    # 2. Extract constants from remote code
    endpoint = remote_ns.get("GRAPHQL_ENDPOINT")
    headers = remote_ns.get("HEADERS")
    api_base_url = remote_ns.get("API_BASE_URL")

    if not endpoint or not headers or not api_base_url:
        raise ConfigEntryNotReady("License validation succeeded but remote configuration is missing.")

    hass.data.setdefault(DOMAIN, {})
    
    email = entry.data.get(CONF_EMAIL)
    password = entry.data.get(CONF_PASSWORD)
    
    # 3. Initialize API with remote constants
    api = ZingMusicAPI(
            email=email, 
            password=password, 
            endpoint=endpoint, 
            headers=headers, 
            api_base_url=api_base_url
        )
        
    try:
        # Perform authentication
        await hass.async_add_executor_job(api.authenticate)
    except ZingMusicUserError as e:
        # If auth fails, raise AuthFailed to prompt re-auth flow (or in this case, user might need to fix via options)
        raise ConfigEntryAuthFailed(f"Authentication failed: {e}")
    except Exception as e:
        raise ConfigEntryNotReady(f"Failed to connect: {e}")

    hass.data[DOMAIN][entry.entry_id] = {
        "api": api,
        "allowed_entity_ids": entry.options.get(CONF_ALLOWED_ENTITY_IDS),
    }

    # Forward to platforms
    await hass.config_entries.async_forward_entry_setups(entry, PLATFORMS)

    # Live options update (no reload needed)
    async def _update_listener(hass: HomeAssistant, updated_entry: ConfigEntry):
        store = hass.data[DOMAIN].get(updated_entry.entry_id, {})
        store["allowed_entity_ids"] = updated_entry.options.get(CONF_ALLOWED_ENTITY_IDS)
        async_dispatcher_send(hass, f"{SIGNAL_OPTIONS_UPDATED}_{updated_entry.entry_id}")

    entry.async_on_unload(entry.add_update_listener(_update_listener))
    return True


async def async_unload_entry(hass: HomeAssistant, entry: ConfigEntry):
    unload_ok = await hass.config_entries.async_unload_platforms(entry, PLATFORMS)
    if unload_ok:
        hass.data[DOMAIN].pop(entry.entry_id, None)
    return unload_ok