from __future__ import annotations

from datetime import datetime
import logging
from typing import Any

from homeassistant.components.sensor import SensorEntity, SensorDeviceClass
from homeassistant.config_entries import ConfigEntry
from homeassistant.const import EntityCategory
from homeassistant.core import HomeAssistant
from homeassistant.helpers.entity_platform import AddEntitiesCallback
from homeassistant.helpers.update_coordinator import CoordinatorEntity
from homeassistant.util import dt as dt_util

from .const import DOMAIN, CONF_LICENSE_ID, KEY_VALID_UNTIL, KEY_HA_INTEGRATION, KEY_STATUS

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(
    hass: HomeAssistant,
    entry: ConfigEntry,
    async_add_entities: AddEntitiesCallback,
) -> None:
    coordinator = hass.data[DOMAIN][entry.entry_id]
    license_id = entry.data[CONF_LICENSE_ID]
    entities = [
        OnOffExpirySensor(coordinator, license_id),
        OnOffIntegrationSensor(coordinator, license_id),
        OnOffStatusSensor(coordinator, license_id),
        OnOffErrorsSensor(coordinator, license_id),
        OnOffSupportEmailSensor(coordinator, license_id),
        OnOffBillingEmailSensor(coordinator, license_id),
        OnOffSupportHotlineSensor(coordinator, license_id),
        OnOffWebsiteSensor(coordinator, license_id),
    ]
    async_add_entities(entities)


class _Base(CoordinatorEntity, SensorEntity):
    def __init__(self, coordinator, license_id: str, kind: str) -> None:
        super().__init__(coordinator)
        self._license_id = license_id
        self._kind = kind
        slug = license_id.lower().replace('-', '_')
        self._attr_unique_id = f"onoff_licenser_{slug}_{kind}"
        self._attr_device_info = {
            "identifiers": {(DOMAIN, license_id)},
            "name": f"OnOff Licenser {license_id}",
            "manufacturer": "OnOff Automations",
        }


class OnOffExpirySensor(_Base):
    _attr_device_class = SensorDeviceClass.TIMESTAMP

    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "expiry")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_expiry"
        self._attr_name = f"Expiry"
        self._attr_icon = "mdi:calendar-clock"
        self._attr_entity_category = EntityCategory.DIAGNOSTIC

    @property
    def name(self) -> str:
        return f"{self._license_id} expiry"

    @property
    def unique_id(self) -> str:
        return f"{DOMAIN}_{self._license_id}_expiry"

    @property
    def native_value(self) -> Any:
        data = self.coordinator.data or {}
        iso_str = data.get(KEY_VALID_UNTIL)
        if not iso_str:
            return None
        dt = dt_util.parse_datetime(iso_str)
        if dt is None:
            return None
        if dt.tzinfo is None:
            dt = dt_util.as_utc(dt)
        return dt


class OnOffIntegrationSensor(_Base):
    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "integration")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_integration"
        self._attr_name = f"Integration"
        self._attr_icon = "mdi:puzzle"
        self._attr_entity_category = EntityCategory.DIAGNOSTIC

    @property
    def native_value(self):
        data = self.coordinator.data or {}
        return data.get(KEY_HA_INTEGRATION) or "Unknown"

    @property
    def extra_state_attributes(self) -> dict:
        """Return additional attributes."""
        if not self.coordinator.data:
            return {}
        
        return {
            "license_id": self._license_id
        }
        
class OnOffStatusSensor(_Base):
    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "status")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_status"
        self._attr_name = f"Status"
        self._attr_icon = "mdi:shield-check"
        self._attr_entity_category = EntityCategory.DIAGNOSTIC

    @property
    def native_value(self):
        data = self.coordinator.data or {}
        return data.get(KEY_STATUS)


class OnOffErrorsSensor(_Base):
    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "error")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_error"
        self._attr_name = f"Error"
        self._attr_icon = "mdi:alert-circle"
        self._attr_entity_category = EntityCategory.DIAGNOSTIC

    @property
    def native_value(self):
        data = self.coordinator.data or {}
        return data.get("error")


class OnOffSupportEmailSensor(_Base):

    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "support_email")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_support_email"
        self._attr_name = "Support Email"
        self._attr_icon = "mdi:email"

    @property
    def native_value(self):
        return "support@onoffautomations.com"


class OnOffBillingEmailSensor(_Base):

    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "billing_email")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_billing_email"
        self._attr_name = "Billing Email"
        self._attr_icon = "mdi:email-check"

    @property
    def native_value(self):
        return "billing@onoffautomations.com"


class OnOffSupportHotlineSensor(_Base):

    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "support_hotline")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_support_hotline"
        self._attr_name = "Support Hotline"
        self._attr_icon = "mdi:phone"

    @property
    def native_value(self):
        return "(845) 400-8199"


class OnOffWebsiteSensor(_Base):

    def __init__(self, coordinator, license_id: str) -> None:
        super().__init__(coordinator, license_id, "website")
        self.entity_id = f"sensor.onoff_licenser_{license_id}_website"
        self._attr_name = "Website"
        self._attr_icon = "mdi:web"

    @property
    def native_value(self):
        return "onoffautomations.com"
