from __future__ import annotations

from typing import Any, Dict
import voluptuous as vol
import logging

from homeassistant import config_entries
from homeassistant.core import HomeAssistant
from homeassistant.data_entry_flow import FlowResult

from .const import (
    DOMAIN,
    CONF_LICENSE_ID,
    CONF_ACTIVATION_CODE,
    CONF_OWNER_EMAIL,
    KEY_UPDATE_INTERVAL_HOURS,
    DATA_UPDATE_INTERVAL_HOURS,
)
from .licenser import api_register, persist_device_token, save_state

_LOGGER = logging.getLogger(__name__)


class OnOffConfigFlow(config_entries.ConfigFlow, domain=DOMAIN):
    VERSION = 1

    async def async_step_user(self, user_input: Dict[str, Any] | None = None) -> FlowResult:
        errors: Dict[str, str] = {}

        if user_input is not None:
            license_id = user_input[CONF_LICENSE_ID].strip()
            activation_code = user_input[CONF_ACTIVATION_CODE].strip()
            owner_email = user_input[CONF_OWNER_EMAIL].strip().lower()

            for entry in self._async_current_entries():
                if entry.data.get(CONF_LICENSE_ID) == license_id:
                    return self.async_abort(reason="already_configured")

            hass: HomeAssistant = self.hass
            try:
                reg = await api_register(
                    hass,
                    license_id=license_id,
                    activation_code=activation_code,
                    owner_email=owner_email,
                )

                # Expect update interval hours in registration response
                upd = reg.get(KEY_UPDATE_INTERVAL_HOURS)
                if not isinstance(upd, (int, float)):
                    raise ValueError("missing_updateIntervalHours")

                device_token = reg.get("device_token")
                if not device_token:
                    raise ValueError("device_token_missing")

                await persist_device_token(hass, license_id, device_token)
                # Persist initial state including update interval hours
                await save_state(hass, license_id, {DATA_UPDATE_INTERVAL_HOURS: float(upd)})

                # Log success to Home Assistant log
                try:
                    _LOGGER.info(
                        "OnOff Licenser registration succeeded: license_id=%s, updateIntervalHours=%s",
                        license_id,
                        upd,
                    )
                except Exception:
                    # logging should not block the flow
                    pass

                await self.async_set_unique_id(license_id)
                self._abort_if_unique_id_configured()
                return self.async_create_entry(
                    title=f"OnOff License {license_id}",
                    data={
                        CONF_LICENSE_ID: license_id,
                        CONF_OWNER_EMAIL: owner_email,
                    },
                )
            except Exception as e:
                _LOGGER.warning("OnOff registration failed: %s", e)
                errors["base"] = "registration_failed"

        schema = vol.Schema({
            vol.Required(CONF_LICENSE_ID): str,
            vol.Required(CONF_ACTIVATION_CODE): str,
            vol.Required(CONF_OWNER_EMAIL): str,
        })
        return self.async_show_form(step_id="user", data_schema=schema, errors=errors)
