from __future__ import annotations

import logging
from typing import Any

from homeassistant.config_entries import ConfigEntry
from homeassistant.core import HomeAssistant

from .const import (
    DOMAIN,
    PLATFORMS,
    CONF_LICENSE_ID,
    KEY_VALID_UNTIL,
    KEY_HA_INTEGRATION,
    KEY_STATUS,
    DATA_UPDATE_INTERVAL_HOURS,
)
from .licenser import load_state, OnOffCoordinator

_LOGGER = logging.getLogger(__name__)


async def async_setup_entry(hass: HomeAssistant, entry: ConfigEntry) -> bool:
    license_id = entry.data[CONF_LICENSE_ID]
    # Load dynamic update interval from state (must be set at registration time)
    state = await load_state(hass, license_id)
    hours = state.get(DATA_UPDATE_INTERVAL_HOURS)
    if not isinstance(hours, (int, float)) or hours <= 0:
        raise ValueError("missing_update_interval_hours")

    coordinator = OnOffCoordinator(hass, license_id, float(hours))
    await coordinator.async_config_entry_first_refresh()

    hass.data.setdefault(DOMAIN, {})
    hass.data[DOMAIN][entry.entry_id] = coordinator

    await hass.config_entries.async_forward_entry_setups(entry, PLATFORMS)
    return True


async def async_unload_entry(hass: HomeAssistant, entry: ConfigEntry) -> bool:
    if unload_ok := await hass.config_entries.async_unload_platforms(entry, PLATFORMS):
        hass.data[DOMAIN].pop(entry.entry_id, None)
    return unload_ok


def get_license_code(hass: HomeAssistant, license_id: str) -> str | None:
    """Get code for a specific license. Used by other integrations."""
    _LOGGER.debug("get_license_code called for license_id: %s", license_id)
    licenser_data = hass.data.get(DOMAIN, {})
    _LOGGER.debug("Available licenser data keys: %s", list(licenser_data.keys()))
    
    for entry_id, coordinator in licenser_data.items():
        _LOGGER.debug(
            "Checking coordinator for entry %s, has license_id: %s",
            entry_id,
            getattr(coordinator, "license_id", "none"),
        )
        if (
            getattr(coordinator, "license_id", None) == license_id
            and hasattr(coordinator, "get_received_code")
        ):
            code = coordinator.get_received_code()
            _LOGGER.debug(
                "Found matching license %s, code available: %s",
                license_id,
                bool(code),
            )
            return code
    
    _LOGGER.debug("No matching license found for: %s", license_id)
    return None


def is_license_valid(hass: HomeAssistant, license_id: str) -> bool:
    """Return True if the given license_id is currently valid.

    Logic:
    - Find the licenser coordinator for this license_id
    - Look at its DataUpdateCoordinator.data
    - Treat license as valid only if:
      * data["ok"] is True
      * status is one of ("active", "ok", "valid")
      * valid_until (if present) is in the future
    - If anything looks wrong or missing -> False
    """
    _LOGGER.debug("is_license_valid called for license_id: %s", license_id)

    licenser_data = hass.data.get(DOMAIN, {})
    if not licenser_data:
        _LOGGER.warning(
            "No licenser data found in hass.data for domain '%s' when checking %s",
            DOMAIN,
            license_id,
        )
        return False

    # Find coordinator for this license_id
    for entry_id, coordinator in licenser_data.items():
        if getattr(coordinator, "license_id", None) != license_id:
            continue

        data = getattr(coordinator, "data", None) or {}
        _LOGGER.debug(
            "License %s: coordinator entry %s, data: %s",
            license_id,
            entry_id,
            data,
        )

        ok = data.get("ok", False)
        status = data.get(KEY_STATUS)
        valid_until_str = data.get(KEY_VALID_UNTIL)

        # Basic ok flag
        if not ok:
            _LOGGER.warning(
                "License %s considered invalid: data.ok is False (entry %s)",
                license_id,
                entry_id,
            )
            return False

        # Status check
        if status:
            status_norm = str(status).lower()
            if status_norm not in ("active", "ok", "valid"):
                _LOGGER.warning(
                    "License %s considered invalid due to status '%s' (entry %s)",
                    license_id,
                    status,
                    entry_id,
                )
                return False

        # # valid_until check (if present)
        # if valid_until_str:
        #     try:
        #         dt_valid_until = dt_util.parse_datetime(valid_until_str)
        #     except Exception as e:
        #         _LOGGER.warning(
        #             "Failed to parse valid_until '%s' for license %s: %s",
        #             valid_until_str,
        #             license_id,
        #             e,
        #         )
        #         dt_valid_until = None

        #     if dt_valid_until is not None:
        #         now = dt_util.utcnow()
        #         if dt_valid_until < now:
        #             _LOGGER.warning(
        #                 "License %s expired at %s (now: %s)",
        #                 license_id,
        #                 dt_valid_until,
        #                 now,
        #             )
        #             return False

        # If we got here, this coordinator says it's valid
        _LOGGER.debug("License %s is considered valid", license_id)
        return True

    # No coordinator for this license_id – treat as invalid
    _LOGGER.warning(
        "No coordinator found for license %s in onoff_licenser; treating as invalid",
        license_id,
    )
    return False